'use client'

import { useState, useEffect } from 'react'
import { useRouter } from 'next/navigation'
import { supabase, getCurrentUserProfile, isAdmin } from '@/lib/supabase'
import { Cliente } from '@/lib/types'
import Modal from '@/app/components/Modal'

export default function NuevoUsuarioPage() {
  const router = useRouter()
  const [loading, setLoading] = useState(true)
  const [guardando, setGuardando] = useState(false)
  const [clientes, setClientes] = useState<Cliente[]>([])
  const [formData, setFormData] = useState({
    email: '',
    password: '',
    nombre: '',
    cliente_id: '',
  })
  const [modalMensaje, setModalMensaje] = useState<{ isOpen: boolean; mensaje: string; tipo: 'success' | 'error' }>({
    isOpen: false,
    mensaje: '',
    tipo: 'success'
  })

  useEffect(() => {
    checkAuth()
    fetchClientes()
  }, [])

  async function checkAuth() {
    const admin = await isAdmin()
    if (!admin) {
      router.push('/login')
      return
    }
    setLoading(false)
  }

  async function fetchClientes() {
    try {
      const { data, error } = await supabase
        .from('clientes')
        .select('*')
        .eq('activo', true)
        .order('nombre')

      if (error) throw error
      setClientes(data || [])
    } catch (error) {
      console.error('Error fetching clientes:', error)
    }
  }

  async function handleSubmit(e: React.FormEvent) {
    e.preventDefault()
    setGuardando(true)

    try {
      // Nota: Para crear usuarios desde el frontend, necesitas usar una función Edge Function
      // o crear el usuario manualmente en Supabase Dashboard y luego asociarlo aquí.
      // Por ahora, mostraremos instrucciones al admin.

      // Alternativa: Crear el usuario directamente en la tabla usuarios
      // y el admin debe crear la cuenta en Supabase Auth manualmente
      // O usar una función Edge Function de Supabase

      // Por simplicidad, vamos a crear solo el registro en usuarios
      // y el admin debe crear el usuario en Supabase Auth Dashboard
      // o usar una función Edge Function

      setModalMensaje({
        isOpen: true,
        mensaje: 'Para crear usuarios clientes, necesitas usar el Dashboard de Supabase o configurar una Edge Function. Por ahora, crea el usuario manualmente en Supabase Auth y luego asócialo con el cliente.',
        tipo: 'error'
      })

      // TODO: Implementar con Edge Function o usar Supabase Admin API desde el backend
    } catch (error: any) {
      console.error('Error creating usuario:', error)
      setModalMensaje({
        isOpen: true,
        mensaje: `Error: ${error.message}`,
        tipo: 'error'
      })
    } finally {
      setGuardando(false)
    }
  }

  if (loading) {
    return (
      <div className="text-center p-8">
        <div className="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        <p className="mt-2">Cargando...</p>
      </div>
    )
  }

  return (
    <div className="max-w-2xl mx-auto">
      <h1 className="text-3xl font-bold mb-6">Crear Usuario Cliente</h1>
      <form onSubmit={handleSubmit} className="bg-white p-6 rounded-lg shadow space-y-4">
        <div>
          <label className="block text-sm font-medium mb-1">Cliente *</label>
          <select
            required
            value={formData.cliente_id}
            onChange={(e) => setFormData({ ...formData, cliente_id: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="">Selecciona un cliente</option>
            {clientes.map((cliente) => (
              <option key={cliente.id} value={cliente.id}>
                {cliente.nombre || cliente.empresa || 'Cliente sin nombre'} {cliente.nombre && cliente.empresa ? `- ${cliente.empresa}` : ''}
              </option>
            ))}
          </select>
          <p className="text-xs text-gray-500 mt-1">Selecciona el cliente al que se asociará este usuario</p>
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Email *</label>
          <input
            type="email"
            required
            value={formData.email}
            onChange={(e) => setFormData({ ...formData, email: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder="cliente@ejemplo.com"
          />
          <p className="text-xs text-gray-500 mt-1">El cliente usará este email para iniciar sesión</p>
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Contraseña *</label>
          <input
            type="password"
            required
            minLength={6}
            value={formData.password}
            onChange={(e) => setFormData({ ...formData, password: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder="Mínimo 6 caracteres"
          />
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Nombre (Opcional)</label>
          <input
            type="text"
            value={formData.nombre}
            onChange={(e) => setFormData({ ...formData, nombre: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder="Nombre del usuario"
          />
        </div>

        <div className="flex gap-4 pt-4">
          <button
            type="submit"
            disabled={guardando}
            className="bg-blue-600 text-white px-6 py-2 rounded hover:bg-blue-700 disabled:opacity-50 transition"
          >
            {guardando ? 'Creando...' : 'Crear Usuario Cliente'}
          </button>
          <button
            type="button"
            onClick={() => router.back()}
            className="bg-gray-300 text-gray-700 px-6 py-2 rounded hover:bg-gray-400 transition"
          >
            Cancelar
          </button>
        </div>
      </form>

      {/* Modal de mensaje */}
      <Modal
        isOpen={modalMensaje.isOpen}
        onClose={() => setModalMensaje({ isOpen: false, mensaje: '', tipo: 'success' })}
        title={modalMensaje.tipo === 'success' ? 'Éxito' : 'Error'}
        message={modalMensaje.mensaje}
        type={modalMensaje.tipo}
      />
    </div>
  )
}
