// API Route para consultar ruts.info desde el servidor (evita problemas de CORS)
import { NextRequest, NextResponse } from 'next/server'

export async function GET(request: NextRequest) {
  try {
    const searchParams = request.nextUrl.searchParams
    const rut = searchParams.get('rut')
    
    // Obtener API key de variable de entorno
    const apiKey = process.env.RUTS_INFO_API_KEY

    if (!rut) {
      return NextResponse.json(
        { error: 'RUT es requerido' },
        { status: 400 }
      )
    }

    if (!apiKey) {
      return NextResponse.json(
        { error: 'API Key de ruts.info no configurada. Agrega RUTS_INFO_API_KEY en .env.local' },
        { status: 500 }
      )
    }

    // Limpiar el RUT (remover puntos, espacios y guión)
    const rutLimpio = rut.replace(/\./g, '').replace(/\s/g, '').replace(/-/g, '').trim()
    
    // Validar formato básico de RUT chileno (7-9 dígitos)
    if (!/^\d{7,9}$/.test(rutLimpio)) {
      return NextResponse.json(
        { error: 'Formato de RUT inválido' },
        { status: 400 }
      )
    }

    // Hacer la petición a ruts.info desde el servidor
    const response = await fetch(`https://ruts.info/api/company-info?rut=${rutLimpio}`, {
      method: 'GET',
      headers: {
        'x-api-key': apiKey,
        'Accept': 'application/json',
      },
    })

    if (!response.ok) {
      const errorText = await response.text()
      let errorData
      try {
        errorData = JSON.parse(errorText)
      } catch {
        errorData = { message: errorText }
      }
      
      return NextResponse.json(
        { error: `Error en ruts.info: ${response.status} ${response.statusText}`, details: errorData },
        { status: response.status }
      )
    }

    const data = await response.json()

    return NextResponse.json(data)
  } catch (error: any) {
    console.error('Error en API route ruts.info:', error)
    return NextResponse.json(
      { error: 'Error al consultar ruts.info', message: error.message },
      { status: 500 }
    )
  }
}
