// API Route para consultar simpleapi.cl desde el servidor (evita problemas de CORS)
import { NextRequest, NextResponse } from 'next/server'

export async function GET(request: NextRequest) {
  try {
    const searchParams = request.nextUrl.searchParams
    const rut = searchParams.get('rut')
    
    // Obtener API key de variable de entorno (opcional para plan gratuito)
    const apiKey = process.env.SIMPLEAPI_KEY

    if (!rut) {
      return NextResponse.json(
        { error: 'RUT es requerido' },
        { status: 400 }
      )
    }

    // Limpiar el RUT (remover puntos, espacios y guión)
    const rutLimpio = rut.replace(/\./g, '').replace(/\s/g, '').replace(/-/g, '').trim()
    
    // Validar formato básico de RUT chileno (7-9 dígitos)
    if (!/^\d{7,9}$/.test(rutLimpio)) {
      return NextResponse.json(
        { error: 'Formato de RUT inválido' },
        { status: 400 }
      )
    }

    // Construir headers para la petición
    const headers: HeadersInit = {
      'Accept': 'application/json',
    }

    // Si hay API key, agregarla al header
    if (apiKey) {
      headers['x-api-key'] = apiKey
    }

    // Hacer la petición a simpleapi.cl desde el servidor
    // Según documentación de simpleapi.cl, el endpoint puede variar
    // Probamos con el formato más común
    let apiUrl = `https://api.simpleapi.cl/api/rut/${rutLimpio}`
    
    // Log para debugging
    console.log('Consultando simpleapi.cl:', {
      url: apiUrl,
      rut: rutLimpio,
      hasApiKey: !!apiKey,
      apiKey: apiKey ? `${apiKey.substring(0, 10)}...` : 'no configurada'
    })
    
    const response = await fetch(apiUrl, {
      method: 'GET',
      headers: headers,
    })
    
    console.log('Respuesta de simpleapi.cl:', {
      status: response.status,
      statusText: response.statusText,
      ok: response.ok
    })

    if (!response.ok) {
      const errorText = await response.text()
      let errorData
      try {
        errorData = JSON.parse(errorText)
      } catch {
        errorData = { message: errorText }
      }
      
      // Si es 401, puede ser que necesite API key o se agotaron las consultas gratuitas
      if (response.status === 401) {
        return NextResponse.json(
          { error: 'API key inválida o se agotaron las consultas gratuitas (10/mes). Verifica tu API key en .env.local' },
          { status: 401 }
        )
      }
      
      return NextResponse.json(
        { error: `Error en simpleapi.cl: ${response.status} ${response.statusText}`, details: errorData },
        { status: response.status }
      )
    }

    const data = await response.json()

    return NextResponse.json(data)
  } catch (error: any) {
    console.error('Error en API route simpleapi.cl:', error)
    return NextResponse.json(
      { error: 'Error al consultar simpleapi.cl', message: error.message },
      { status: 500 }
    )
  }
}
