'use client'

import { useEffect, useState } from 'react'
import { useRouter } from 'next/navigation'
import { supabase, getCurrentUserProfile } from '@/lib/supabase'
import { Usuario, Cliente, Suscripcion, Pago } from '@/lib/types'
import Link from 'next/link'
import Modal from '@/app/components/Modal'

export default function PerfilClientePage() {
  const router = useRouter()
  const [loading, setLoading] = useState(true)
  const [usuario, setUsuario] = useState<Usuario | null>(null)
  const [cliente, setCliente] = useState<Cliente | null>(null)
  const [suscripciones, setSuscripciones] = useState<Suscripcion[]>([])
  const [pagos, setPagos] = useState<Pago[]>([])
  const [modalMensaje, setModalMensaje] = useState<{ isOpen: boolean; mensaje: string; tipo: 'success' | 'error' }>({
    isOpen: false,
    mensaje: '',
    tipo: 'success'
  })

  useEffect(() => {
    checkAuthAndLoadData()
  }, [])

  async function checkAuthAndLoadData() {
    try {
      const { data: { user } } = await supabase.auth.getUser()
      
      if (!user) {
        router.push('/login')
        return
      }

      // Obtener perfil del usuario
      const profile = await getCurrentUserProfile()
      if (!profile) {
        router.push('/login')
        return
      }

      if (profile.rol !== 'cliente') {
        router.push('/')
        return
      }

      setUsuario(profile as Usuario)

      // Si tiene cliente_id, cargar datos del cliente
      if (profile.cliente_id) {
        await loadClienteData(profile.cliente_id)
      }
    } catch (error) {
      console.error('Error checking auth:', error)
      router.push('/login')
    } finally {
      setLoading(false)
    }
  }

  async function loadClienteData(clienteId: string) {
    try {
      // Cargar datos del cliente
      const { data: clienteData, error: clienteError } = await supabase
        .from('clientes')
        .select('*')
        .eq('id', clienteId)
        .single()

      if (clienteError) throw clienteError
      setCliente(clienteData)

      // Cargar suscripciones del cliente
      const { data: suscripcionesData, error: suscripcionesError } = await supabase
        .from('suscripciones')
        .select('*')
        .eq('cliente_id', clienteId)
        .order('created_at', { ascending: false })

      if (suscripcionesError) throw suscripcionesError
      setSuscripciones(suscripcionesData || [])

      // Cargar pagos del cliente
      const { data: pagosData, error: pagosError } = await supabase
        .from('pagos')
        .select('*')
        .eq('cliente_id', clienteId)
        .order('created_at', { ascending: false })
        .limit(10)

      if (pagosError) throw pagosError
      setPagos(pagosData || [])
    } catch (error) {
      console.error('Error loading cliente data:', error)
      setModalMensaje({
        isOpen: true,
        mensaje: 'Error al cargar los datos. Por favor, intenta nuevamente.',
        tipo: 'error'
      })
    }
  }

  async function handleLogout() {
    await supabase.auth.signOut()
    router.push('/login')
    router.refresh()
  }

  function estaVencida(fechaFin?: string): boolean {
    if (!fechaFin) return false
    const fechaStr = fechaFin.split('T')[0]
    const hoyStr = new Date().toISOString().split('T')[0]
    return fechaStr < hoyStr
  }

  function estaPorVencer(fechaFin?: string): boolean {
    if (!fechaFin) return false
    const fechaStr = fechaFin.split('T')[0]
    const hoyStr = new Date().toISOString().split('T')[0]
    const [year, month, day] = fechaStr.split('-').map(Number)
    const [yearHoy, monthHoy, dayHoy] = hoyStr.split('-').map(Number)
    const fecha = new Date(year, month - 1, day)
    const hoy = new Date(yearHoy, monthHoy - 1, dayHoy)
    const diasRestantes = Math.ceil((fecha.getTime() - hoy.getTime()) / (1000 * 60 * 60 * 24))
    return diasRestantes <= 7 && diasRestantes > 0
  }

  if (loading) {
    return (
      <div className="text-center p-8">
        <div className="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        <p className="mt-2">Cargando...</p>
      </div>
    )
  }

  if (!usuario || !cliente) {
    return (
      <div className="text-center p-8">
        <p className="text-gray-500">No se encontraron datos del cliente.</p>
      </div>
    )
  }

  return (
    <div className="max-w-6xl mx-auto">
      {/* Header */}
      <div className="bg-white p-6 rounded-lg shadow mb-6">
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-3xl font-bold">Mi Perfil</h1>
            <p className="text-gray-600 mt-1">{cliente.nombre || cliente.empresa || 'Cliente'}</p>
          </div>
          <button
            onClick={handleLogout}
            className="bg-red-600 text-white px-4 py-2 rounded hover:bg-red-700 transition"
          >
            Cerrar Sesión
          </button>
        </div>
      </div>

      {/* Información del Cliente */}
      <div className="bg-white p-6 rounded-lg shadow mb-6">
        <h2 className="text-2xl font-bold mb-4">Información Personal</h2>
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          {cliente.nombre && (
            <div>
              <p className="text-sm text-gray-600">Nombre</p>
              <p className="font-semibold">{cliente.nombre}</p>
            </div>
          )}
          {cliente.empresa && (
            <div>
              <p className="text-sm text-gray-600">Empresa</p>
              <p className="font-semibold">{cliente.empresa}</p>
            </div>
          )}
          {cliente.email && (
            <div>
              <p className="text-sm text-gray-600">Email</p>
              <p className="font-semibold">{cliente.email}</p>
            </div>
          )}
          {cliente.telefono && (
            <div>
              <p className="text-sm text-gray-600">Teléfono</p>
              <p className="font-semibold">{cliente.telefono}</p>
            </div>
          )}
          {cliente.rfc && (
            <div>
              <p className="text-sm text-gray-600">RUT</p>
              <p className="font-semibold">{cliente.rfc}</p>
            </div>
          )}
          {cliente.direccion && (
            <div className="md:col-span-2">
              <p className="text-sm text-gray-600">Dirección</p>
              <p className="font-semibold">{cliente.direccion}</p>
            </div>
          )}
        </div>
      </div>

      {/* Suscripciones */}
      <div className="bg-white p-6 rounded-lg shadow mb-6">
        <h2 className="text-2xl font-bold mb-4">Mis Suscripciones</h2>
        {suscripciones.length === 0 ? (
          <p className="text-gray-500">No tienes suscripciones activas.</p>
        ) : (
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            {suscripciones.map((suscripcion) => {
              const vencida = estaVencida(suscripcion.fecha_fin)
              const porVencer = estaPorVencer(suscripcion.fecha_fin)
              
              return (
                <div
                  key={suscripcion.id}
                  className={`p-4 rounded-lg border-l-4 ${
                    vencida ? 'bg-red-50 border-red-300' :
                    porVencer ? 'bg-yellow-50 border-yellow-300' :
                    'bg-green-50 border-green-300'
                  }`}
                >
                  <div className="flex justify-between items-start mb-2">
                    <div>
                      {suscripcion.patente && (
                        <p className="font-bold text-lg mb-1">🚗 {suscripcion.patente}</p>
                      )}
                      <p className="text-sm text-gray-600">
                        {suscripcion.tipo === 'anual' ? '📅 Anual' : '📆 Mensual'}
                      </p>
                    </div>
                    <span className={`px-2 py-1 rounded text-xs ${
                      vencida ? 'bg-red-100 text-red-800' :
                      porVencer ? 'bg-yellow-100 text-yellow-800' :
                      'bg-green-100 text-green-800'
                    }`}>
                      {vencida ? 'Vencida' : porVencer ? 'Por vencer' : 'Activa'}
                    </span>
                  </div>
                  <p className="text-xl font-bold text-blue-600 mb-2">
                    ${suscripcion.precio.toLocaleString()}
                  </p>
                  <p className="text-xs text-gray-500">
                    Inicio: {(() => {
                      const fecha = suscripcion.fecha_inicio.split('T')[0]
                      const [year, month, day] = fecha.split('-')
                      return `${day}/${month}/${year}`
                    })()}
                  </p>
                  {suscripcion.fecha_fin && (
                    <p className="text-xs text-gray-500">
                      Fin: {(() => {
                        const fecha = suscripcion.fecha_fin.split('T')[0]
                        const [year, month, day] = fecha.split('-')
                        return `${day}/${month}/${year}`
                      })()}
                    </p>
                  )}
                </div>
              )
            })}
          </div>
        )}
      </div>

      {/* Pagos Recientes */}
      <div className="bg-white p-6 rounded-lg shadow">
        <h2 className="text-2xl font-bold mb-4">Mis Pagos Recientes</h2>
        {pagos.length === 0 ? (
          <p className="text-gray-500">No tienes pagos registrados.</p>
        ) : (
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gray-50">
                <tr>
                  <th className="px-4 py-3 text-left text-sm font-semibold">Monto</th>
                  <th className="px-4 py-3 text-left text-sm font-semibold">Fecha</th>
                  <th className="px-4 py-3 text-left text-sm font-semibold">Método</th>
                  <th className="px-4 py-3 text-left text-sm font-semibold">Estado</th>
                  <th className="px-4 py-3 text-left text-sm font-semibold">Acciones</th>
                </tr>
              </thead>
              <tbody>
                {pagos.map((pago) => (
                  <tr key={pago.id} className="border-t">
                    <td className="px-4 py-3 font-semibold">${pago.monto.toLocaleString()}</td>
                    <td className="px-4 py-3">
                      {(() => {
                        const fecha = pago.fecha_pago.split('T')[0]
                        const [year, month, day] = fecha.split('-')
                        return `${day}/${month}/${year}`
                      })()}
                    </td>
                    <td className="px-4 py-3">{pago.metodo_pago || '-'}</td>
                    <td className="px-4 py-3">
                      <span className={`px-2 py-1 rounded text-xs ${
                        pago.estado === 'completado' ? 'bg-green-100 text-green-800' :
                        pago.estado === 'cancelado' ? 'bg-gray-100 text-gray-800' :
                        'bg-yellow-100 text-yellow-800'
                      }`}>
                        {pago.estado === 'completado' ? 'Completado' :
                         pago.estado === 'cancelado' ? 'Cancelado' :
                         'Pendiente'}
                      </span>
                    </td>
                    <td className="px-4 py-3">
                      <Link
                        href={`/pagos/${pago.id}/comprobante`}
                        className="text-blue-600 hover:underline text-sm"
                      >
                        Ver Comprobante
                      </Link>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
      </div>

      {/* Modal de mensaje */}
      <Modal
        isOpen={modalMensaje.isOpen}
        onClose={() => setModalMensaje({ isOpen: false, mensaje: '', tipo: 'success' })}
        title={modalMensaje.tipo === 'success' ? 'Éxito' : 'Error'}
        message={modalMensaje.mensaje}
        type={modalMensaje.tipo}
      />
    </div>
  )
}
