'use client'

import { useState } from 'react'
import { useRouter } from 'next/navigation'
import { supabase } from '@/lib/supabase'
import { consultarEmpresaSimpleApi, formatearDireccionSimpleApi, obtenerNombreEmpresa, SimpleApiCompany } from '@/lib/simpleapi-api'
import Modal from '@/app/components/Modal'

export default function NuevoClientePage() {
  const router = useRouter()
  const [tipoCliente, setTipoCliente] = useState<'persona' | 'empresa'>('persona')
  const [loading, setLoading] = useState(false)
  const [consultandoRUT, setConsultandoRUT] = useState(false)
  const [modalMensaje, setModalMensaje] = useState<{ isOpen: boolean; mensaje: string; tipo: 'success' | 'error' }>({
    isOpen: false,
    mensaje: '',
    tipo: 'success'
  })
  const [formData, setFormData] = useState({
    nombre: '',
    email: '',
    telefono: '',
    empresa: '',
    direccion: '',
    rfc: '',
    notas: '',
    activo: true,
  })

  async function consultarRUT() {
    if (!formData.rfc || formData.rfc.length < 8) {
      setModalMensaje({
        isOpen: true,
        mensaje: 'Por favor ingresa un RUT válido (mínimo 8 caracteres)',
        tipo: 'error'
      })
      return
    }

    setConsultandoRUT(true)
    try {
      // Consultar empresa desde simpleapi.cl
      const datosEmpresa = await consultarEmpresaSimpleApi(formData.rfc)
      
      if (datosEmpresa) {
        // Autocompletar campos con datos de simpleapi.cl
        setFormData(prev => ({
          ...prev,
          rfc: datosEmpresa.rut || prev.rfc,
          empresa: obtenerNombreEmpresa(datosEmpresa) || prev.empresa,
          email: datosEmpresa.email || prev.email,
          telefono: datosEmpresa.telefono || prev.telefono,
          direccion: formatearDireccionSimpleApi(datosEmpresa) || prev.direccion,
          notas: prev.notas || (datosEmpresa.giro ? `Giro: ${datosEmpresa.giro}` : ''),
        }))
        
        setModalMensaje({
          isOpen: true,
          mensaje: `Datos de ${obtenerNombreEmpresa(datosEmpresa) || 'la empresa'} cargados exitosamente`,
          tipo: 'success'
        })
      } else {
        setModalMensaje({
          isOpen: true,
          mensaje: 'No se encontró información para este RUT en simpleapi.cl. Puedes completar los datos manualmente.',
          tipo: 'error'
        })
      }
    } catch (error: any) {
      console.error('Error consultando simpleapi.cl:', error)
      setModalMensaje({
        isOpen: true,
        mensaje: 'Error al consultar simpleapi.cl. Verifica que tengas configurada la API key en .env.local (opcional para plan gratuito)',
        tipo: 'error'
      })
    } finally {
      setConsultandoRUT(false)
    }
  }

  async function handleSubmit(e: React.FormEvent) {
    e.preventDefault()
    
    // Validar según el tipo de cliente
    if (tipoCliente === 'persona' && !formData.nombre) {
      setModalMensaje({
        isOpen: true,
        mensaje: 'Debes ingresar el nombre de la persona',
        tipo: 'error'
      })
      return
    }
    
    if (tipoCliente === 'empresa' && !formData.empresa) {
      setModalMensaje({
        isOpen: true,
        mensaje: 'Debes ingresar el nombre de la empresa',
        tipo: 'error'
      })
      return
    }
    
    setLoading(true)

    try {
      const { error } = await supabase
        .from('clientes')
        .insert([{
          ...formData,
          nombre: tipoCliente === 'persona' ? formData.nombre : null,
        }])

      if (error) throw error

      router.push('/clientes')
    } catch (error: any) {
      console.error('Error creating cliente:', error)
      setModalMensaje({
        isOpen: true,
        mensaje: `Error al crear el cliente: ${error.message}`,
        tipo: 'error'
      })
    } finally {
      setLoading(false)
    }
  }

  return (
    <div className="max-w-2xl mx-auto">
      <h1 className="text-3xl font-bold mb-6">Nuevo Cliente</h1>
      
      {/* Selector de tipo de cliente */}
      <div className="mb-6 bg-white p-4 rounded-lg shadow">
        <div className="flex gap-4">
          <button
            type="button"
            onClick={() => setTipoCliente('persona')}
            className={`flex-1 px-4 py-3 rounded-lg font-medium transition ${
              tipoCliente === 'persona'
                ? 'bg-blue-600 text-white'
                : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
            }`}
          >
            👤 Persona Natural
          </button>
          <button
            type="button"
            onClick={() => setTipoCliente('empresa')}
            className={`flex-1 px-4 py-3 rounded-lg font-medium transition ${
              tipoCliente === 'empresa'
                ? 'bg-blue-600 text-white'
                : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
            }`}
          >
            🏢 Empresa
          </button>
        </div>
      </div>

      <form onSubmit={handleSubmit} className="bg-white p-6 rounded-lg shadow space-y-4">
        {tipoCliente === 'persona' && (
          <div>
            <label className="block text-sm font-medium mb-1">Nombre *</label>
            <input
              type="text"
              required
              value={formData.nombre}
              onChange={(e) => setFormData({ ...formData, nombre: e.target.value })}
              className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
              placeholder="Nombre completo"
            />
          </div>
        )}
        <div>
          <label className="block text-sm font-medium mb-1">Email</label>
          <input
            type="email"
            value={formData.email}
            onChange={(e) => setFormData({ ...formData, email: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder="correo@ejemplo.com"
          />
        </div>
        <div>
          <label className="block text-sm font-medium mb-1">Teléfono</label>
          <input
            type="tel"
            value={formData.telefono}
            onChange={(e) => setFormData({ ...formData, telefono: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder="+52 123 456 7890"
          />
        </div>
        {tipoCliente === 'empresa' && (
          <>
            <div>
              <label className="block text-sm font-medium mb-1">RUT *</label>
              <div className="flex gap-2">
                <input
                  type="text"
                  required
                  value={formData.rfc}
                  onChange={(e) => setFormData({ ...formData, rfc: e.target.value })}
                  className="flex-1 border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                  placeholder="12345678-9"
                  onBlur={() => {
                    // Auto-consultar cuando se sale del campo si tiene formato válido
                    if (formData.rfc && formData.rfc.length >= 8 && !formData.empresa) {
                      consultarRUT()
                    }
                  }}
                />
                <button
                  type="button"
                  onClick={consultarRUT}
                  disabled={consultandoRUT || !formData.rfc}
                  className="bg-green-600 text-white px-4 py-2 rounded hover:bg-green-700 disabled:opacity-50 disabled:cursor-not-allowed transition"
                  title="Consultar datos desde simpleapi.cl"
                >
                  {consultandoRUT ? '⏳' : '🔍'}
                </button>
              </div>
              <p className="text-xs text-gray-500 mt-1">
                Ingresa el RUT y haz clic en 🔍 para autocompletar los datos desde simpleapi.cl (10 consultas/mes gratis)
              </p>
            </div>
            <div>
              <label className="block text-sm font-medium mb-1">Razón Social / Nombre de Empresa *</label>
              <input
                type="text"
                required
                value={formData.empresa}
                onChange={(e) => setFormData({ ...formData, empresa: e.target.value })}
                className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                placeholder="Nombre de la empresa"
              />
            </div>
          </>
        )}
        
        {tipoCliente === 'persona' && (
          <div>
            <label className="block text-sm font-medium mb-1">RUT</label>
            <input
              type="text"
              value={formData.rfc}
              onChange={(e) => setFormData({ ...formData, rfc: e.target.value })}
              className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
              placeholder="12345678-9"
            />
          </div>
        )}


        <div>
          <label className="block text-sm font-medium mb-1">Dirección</label>
          <textarea
            value={formData.direccion}
            onChange={(e) => setFormData({ ...formData, direccion: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            rows={3}
            placeholder="Dirección completa"
          />
        </div>
        <div>
          <label className="block text-sm font-medium mb-1">Notas</label>
          <textarea
            value={formData.notas}
            onChange={(e) => setFormData({ ...formData, notas: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            rows={3}
            placeholder="Notas adicionales sobre el cliente"
          />
        </div>
        <div className="flex items-center">
          <input
            type="checkbox"
            checked={formData.activo}
            onChange={(e) => setFormData({ ...formData, activo: e.target.checked })}
            className="mr-2"
          />
          <label className="text-sm font-medium">Cliente activo</label>
        </div>
        <div className="flex gap-4 pt-4">
          <button
            type="submit"
            disabled={loading}
            className="bg-blue-600 text-white px-6 py-2 rounded hover:bg-blue-700 disabled:opacity-50 transition"
          >
            {loading ? 'Guardando...' : 'Guardar Cliente'}
          </button>
          <button
            type="button"
            onClick={() => router.back()}
            className="bg-gray-300 text-gray-700 px-6 py-2 rounded hover:bg-gray-400 transition"
          >
            Cancelar
          </button>
        </div>
      </form>

      {/* Modal de mensajes */}
      <Modal
        isOpen={modalMensaje.isOpen}
        onClose={() => setModalMensaje({ isOpen: false, mensaje: '', tipo: 'success' })}
        title={modalMensaje.tipo === 'success' ? 'Éxito' : 'Error'}
        message={modalMensaje.mensaje}
        type={modalMensaje.tipo}
      />
    </div>
  )
}
