'use client'

import { useEffect, useState } from 'react'
import { supabase } from '@/lib/supabase'
import { Cliente } from '@/lib/types'
import Link from 'next/link'
import Modal from '../components/Modal'

export default function ClientesPage() {
  const [clientes, setClientes] = useState<Cliente[]>([])
  const [loading, setLoading] = useState(true)
  const [busqueda, setBusqueda] = useState('')
  const [eliminandoId, setEliminandoId] = useState<string | null>(null)
  const [modalConfirmar, setModalConfirmar] = useState<{ isOpen: boolean; clienteId: string | null }>({
    isOpen: false,
    clienteId: null
  })
  const [modalMensaje, setModalMensaje] = useState<{ isOpen: boolean; mensaje: string; tipo: 'success' | 'error' }>({
    isOpen: false,
    mensaje: '',
    tipo: 'success'
  })

  useEffect(() => {
    fetchClientes()
  }, [])

  async function fetchClientes() {
    try {
      const { data, error } = await supabase
        .from('clientes')
        .select('*')
        .order('created_at', { ascending: false })

      if (error) throw error
      setClientes(data || [])
    } catch (error) {
      console.error('Error fetching clientes:', error)
      setModalMensaje({
        isOpen: true,
        mensaje: 'Error al cargar clientes. Asegúrate de que las tablas estén creadas en Supabase.',
        tipo: 'error'
      })
    } finally {
      setLoading(false)
    }
  }

  function abrirModalEliminar(id: string) {
    setModalConfirmar({ isOpen: true, clienteId: id })
  }

  async function eliminarCliente() {
    if (!modalConfirmar.clienteId) return

    setEliminandoId(modalConfirmar.clienteId)
    setModalConfirmar({ isOpen: false, clienteId: null })

    try {
      const { error } = await supabase
        .from('clientes')
        .delete()
        .eq('id', modalConfirmar.clienteId)

      if (error) throw error

      setModalMensaje({
        isOpen: true,
        mensaje: 'Cliente eliminado exitosamente',
        tipo: 'success'
      })
      fetchClientes()
    } catch (error: any) {
      console.error('Error deleting cliente:', error)
      setModalMensaje({
        isOpen: true,
        mensaje: `Error al eliminar el cliente: ${error.message}`,
        tipo: 'error'
      })
    } finally {
      setEliminandoId(null)
    }
  }

  if (loading) {
    return (
      <div className="text-center p-8">
        <div className="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        <p className="mt-2">Cargando clientes...</p>
      </div>
    )
  }

  return (
    <div>
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-3xl font-bold">Clientes</h1>
        <Link
          href="/clientes/nuevo"
          className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700 transition"
        >
          + Nuevo Cliente
        </Link>
      </div>

      <div className="mb-6">
        <input
          type="text"
          placeholder="🔍 Buscar por nombre, email, empresa o teléfono..."
          value={busqueda}
          onChange={(e) => setBusqueda(e.target.value)}
          className="w-full max-w-md border rounded-lg px-4 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
        />
      </div>

      {(() => {
        const clientesFiltrados = clientes.filter((cliente) => {
          if (!busqueda) return true
          const termino = busqueda.toLowerCase()
          return (
            cliente.nombre?.toLowerCase().includes(termino) ||
            cliente.email?.toLowerCase().includes(termino) ||
            cliente.empresa?.toLowerCase().includes(termino) ||
            cliente.telefono?.toLowerCase().includes(termino) ||
            cliente.rfc?.toLowerCase().includes(termino)
          )
        })

        return (
          <>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              {clientesFiltrados.map((cliente) => (
          <div
            key={cliente.id}
            className="bg-white p-4 rounded-lg shadow hover:shadow-lg transition relative"
          >
            <div className="absolute top-2 right-2 flex gap-2">
              <Link
                href={`/clientes/${cliente.id}/editar`}
                onClick={(e) => e.stopPropagation()}
                className="text-gray-600 hover:text-blue-600 transition p-1"
                title="Editar cliente"
              >
                ✏️
              </Link>
              <button
                onClick={(e) => {
                  e.stopPropagation()
                  abrirModalEliminar(cliente.id)
                }}
                disabled={eliminandoId === cliente.id}
                className="text-gray-600 hover:text-red-600 disabled:opacity-50 transition p-1"
                title="Eliminar cliente"
              >
                {eliminandoId === cliente.id ? '...' : '🗑️'}
              </button>
            </div>
            <Link
              href={`/clientes/${cliente.id}`}
              className="block"
            >
              <h3 className="font-bold text-lg mb-2 pr-12">{cliente.nombre || cliente.empresa || 'Cliente sin nombre'}</h3>
              {cliente.empresa && cliente.nombre && <p className="text-gray-600 text-sm mb-1">🏢 {cliente.empresa}</p>}
              {!cliente.nombre && cliente.empresa && <p className="text-gray-500 text-sm mb-1">Empresa</p>}
              {cliente.email && <p className="text-sm text-gray-500 mb-1">📧 {cliente.email}</p>}
              {cliente.telefono && <p className="text-sm text-gray-500 mb-2">📞 {cliente.telefono}</p>}
              <span className={`inline-block px-2 py-1 rounded text-xs ${
                cliente.activo ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'
              }`}>
                {cliente.activo ? '✓ Activo' : '✗ Inactivo'}
              </span>
            </Link>
          </div>
        ))}
            </div>

            {clientesFiltrados.length === 0 && clientes.length > 0 && (
              <div className="text-center p-8 bg-white rounded-lg shadow mt-4">
                <p className="text-gray-500">No se encontraron clientes que coincidan con "{busqueda}"</p>
              </div>
            )}
          </>
        )
      })()}

      {clientes.length === 0 && (
        <div className="text-center p-8 bg-white rounded-lg shadow">
          <p className="text-gray-500 mb-4">No hay clientes registrados.</p>
          <Link
            href="/clientes/nuevo"
            className="inline-block bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700"
          >
            Crear primer cliente
          </Link>
        </div>
      )}

      {/* Modal de confirmación para eliminar */}
      <Modal
        isOpen={modalConfirmar.isOpen}
        onClose={() => setModalConfirmar({ isOpen: false, clienteId: null })}
        title="Eliminar Cliente"
        message="¿Estás seguro de eliminar este cliente? Esta acción eliminará también todas sus suscripciones, pagos y registros relacionados. Esta acción no se puede deshacer."
        onConfirm={eliminarCliente}
        confirmText="Eliminar"
        cancelText="Cancelar"
        type="warning"
      />

      {/* Modal de mensaje */}
      <Modal
        isOpen={modalMensaje.isOpen}
        onClose={() => setModalMensaje({ isOpen: false, mensaje: '', tipo: 'success' })}
        title={modalMensaje.tipo === 'success' ? 'Éxito' : 'Error'}
        message={modalMensaje.mensaje}
        type={modalMensaje.tipo}
      />
    </div>
  )
}
