'use client'

import { useEffect, useState } from 'react'
import { useRouter, usePathname } from 'next/navigation'
import { supabase, getCurrentUserProfile } from '@/lib/supabase'
import { Usuario } from '@/lib/types'

interface AuthGuardProps {
  children: React.ReactNode
  requireAdmin?: boolean
  requireAuth?: boolean
}

export default function AuthGuard({ children, requireAdmin = false, requireAuth = true }: AuthGuardProps) {
  const router = useRouter()
  const pathname = usePathname()
  const [loading, setLoading] = useState(true)
  const [usuario, setUsuario] = useState<Usuario | null>(null)

  useEffect(() => {
    checkAuth()
  }, [pathname])

  async function checkAuth() {
    // Rutas públicas que no requieren autenticación
    const publicRoutes = ['/login', '/formularios/publico']
    if (publicRoutes.some(route => pathname.startsWith(route))) {
      setLoading(false)
      return
    }

    try {
      const { data: { user } } = await supabase.auth.getUser()
      
      if (!user) {
        if (requireAuth) {
          router.push('/login')
          return
        }
        setLoading(false)
        return
      }

      // Obtener perfil del usuario
      const profile = await getCurrentUserProfile()
      if (!profile) {
        if (requireAuth) {
          router.push('/login')
          return
        }
        setLoading(false)
        return
      }

      setUsuario(profile as Usuario)

      // Verificar si requiere admin
      if (requireAdmin && profile.rol !== 'admin') {
        router.push('/cliente/perfil')
        return
      }

      // Si es cliente, redirigir a su perfil si intenta acceder a rutas admin
      if (profile.rol === 'cliente' && pathname.startsWith('/admin')) {
        router.push('/cliente/perfil')
        return
      }

      // Si es admin, solo bloquear /cliente/perfil (perfil del cliente usuario)
      // Los admins pueden acceder a /clientes (lista de clientes) y otras rutas
      if (profile.rol === 'admin' && pathname.startsWith('/cliente/perfil')) {
        router.push('/')
        return
      }
    } catch (error) {
      console.error('Error checking auth:', error)
      if (requireAuth) {
        router.push('/login')
        return
      }
    } finally {
      setLoading(false)
    }
  }

  if (loading) {
    return (
      <div className="text-center p-8">
        <div className="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        <p className="mt-2">Cargando...</p>
      </div>
    )
  }

  return <>{children}</>
}
