'use client'

import { useState, useEffect, useRef } from 'react'
import { useRouter } from 'next/navigation'
import { supabase, getCurrentUserProfile } from '@/lib/supabase'
import { Usuario } from '@/lib/types'

export default function UserMenu() {
  const router = useRouter()
  const [isOpen, setIsOpen] = useState(false)
  const [usuario, setUsuario] = useState<Usuario | null>(null)
  const menuRef = useRef<HTMLDivElement>(null)

  useEffect(() => {
    loadUser()
    
    // Cerrar el menú al hacer clic fuera
    function handleClickOutside(event: MouseEvent) {
      if (menuRef.current && !menuRef.current.contains(event.target as Node)) {
        setIsOpen(false)
      }
    }

    if (isOpen) {
      document.addEventListener('mousedown', handleClickOutside)
    }

    return () => {
      document.removeEventListener('mousedown', handleClickOutside)
    }
  }, [isOpen])

  async function loadUser() {
    try {
      const profile = await getCurrentUserProfile()
      setUsuario(profile)
    } catch (error) {
      console.error('Error loading user:', error)
    }
  }

  async function handleLogout() {
    try {
      await supabase.auth.signOut()
      router.push('/login')
      router.refresh()
    } catch (error) {
      console.error('Error logging out:', error)
    }
  }

  function handleProfileClick() {
    if (usuario?.rol === 'admin') {
      router.push('/')
    } else {
      router.push('/cliente/perfil')
    }
    setIsOpen(false)
  }

  return (
    <div className="relative" ref={menuRef}>
      <button
        onClick={() => setIsOpen(!isOpen)}
        className="p-2 rounded-md hover:bg-blue-700 transition-colors focus:outline-none focus:ring-2 focus:ring-white"
        aria-label="Perfil de usuario"
        title="Perfil"
      >
        <svg
          className="w-6 h-6"
          fill="none"
          stroke="currentColor"
          viewBox="0 0 24 24"
        >
          <path
            strokeLinecap="round"
            strokeLinejoin="round"
            strokeWidth={2}
            d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"
          />
        </svg>
      </button>

      {isOpen && (
        <div className="absolute right-0 mt-2 w-48 bg-white rounded-md shadow-lg py-1 z-50 border border-gray-200">
          {usuario && (
            <div className="px-4 py-2 border-b border-gray-200">
              <p className="text-sm font-semibold text-gray-900">
                {usuario.nombre || usuario.email}
              </p>
              <p className="text-xs text-gray-500 capitalize">
                {usuario.rol}
              </p>
            </div>
          )}
          <button
            onClick={handleProfileClick}
            className="w-full text-left px-4 py-2 text-sm text-gray-700 hover:bg-gray-100 transition"
          >
            👤 Ver Perfil
          </button>
          <button
            onClick={handleLogout}
            className="w-full text-left px-4 py-2 text-sm text-red-600 hover:bg-red-50 transition"
          >
            🚪 Cerrar Sesión
          </button>
        </div>
      )}
    </div>
  )
}
