'use client'

import { useState, useEffect } from 'react'
import { useRouter } from 'next/navigation'
import { supabase } from '@/lib/supabase'
import { Cliente, Producto } from '@/lib/types'

export default function NuevaCotizacionPage() {
  const router = useRouter()
  const [loading, setLoading] = useState(false)
  const [clientes, setClientes] = useState<Cliente[]>([])
  const [productos, setProductos] = useState<Producto[]>([])
  const [creandoCliente, setCreandoCliente] = useState(false)
  const [nuevoCliente, setNuevoCliente] = useState({
    nombre: '',
    email: '',
    telefono: '',
    empresa: '',
  })
  const [formData, setFormData] = useState({
    cliente_id: '',
    fecha_emision: new Date().toISOString().split('T')[0],
    fecha_vencimiento: '',
    descripcion: '',
    incluir_iva: false,
  })
  const [items, setItems] = useState<Array<{
    producto_id: string
    cantidad: number
    precio_unitario: number
    descripcion: string
    subtotal: number
    incluir_iva: boolean
    iva: number
  }>>([])

  useEffect(() => {
    fetchClientes()
    fetchProductos()
  }, [])

  async function fetchClientes() {
    try {
      const { data, error } = await supabase
        .from('clientes')
        .select('*')
        .eq('activo', true)
        .order('nombre')

      if (error) throw error
      setClientes(data || [])
    } catch (error) {
      console.error('Error fetching clientes:', error)
    }
  }

  async function fetchProductos() {
    try {
      const { data, error } = await supabase
        .from('productos')
        .select('*')
        .eq('activo', true)
        .order('nombre')

      if (error) throw error
      setProductos(data || [])
    } catch (error) {
      console.error('Error fetching productos:', error)
    }
  }

  async function crearClienteRapido() {
    if (!nuevoCliente.nombre && !nuevoCliente.empresa) {
      alert('Debes ingresar al menos un nombre o una empresa')
      return
    }

    try {
      const { data, error } = await supabase
        .from('clientes')
        .insert([{
          nombre: nuevoCliente.nombre || null,
          email: nuevoCliente.email || null,
          telefono: nuevoCliente.telefono || null,
          empresa: nuevoCliente.empresa || null,
          activo: true,
        }])
        .select()
        .single()

      if (error) throw error

      setFormData({ ...formData, cliente_id: data.id })
      setCreandoCliente(false)
      setNuevoCliente({ nombre: '', email: '', telefono: '', empresa: '' })
      await fetchClientes()
      alert('Cliente creado exitosamente')
    } catch (error: any) {
      console.error('Error creating cliente:', error)
      alert(`Error al crear el cliente: ${error.message}`)
    }
  }

  function agregarItem() {
    setItems([...items, {
      producto_id: '',
      cantidad: 1,
      precio_unitario: 0,
      descripcion: '',
      subtotal: 0,
      incluir_iva: false,
      iva: 0,
    }])
  }

  function eliminarItem(index: number) {
    setItems(items.filter((_, i) => i !== index))
  }

  function actualizarItem(index: number, campo: string, valor: any) {
    const nuevosItems = [...items]
    nuevosItems[index] = { ...nuevosItems[index], [campo]: valor }

    // Si cambió el producto, actualizar precio y descripción
    if (campo === 'producto_id') {
      const producto = productos.find(p => p.id === valor)
      if (producto) {
        nuevosItems[index].precio_unitario = producto.precio
        nuevosItems[index].descripcion = producto.descripcion || producto.nombre
      }
    }

    // Calcular subtotal
    nuevosItems[index].subtotal = nuevosItems[index].cantidad * nuevosItems[index].precio_unitario
    
    // Calcular IVA si está marcado (19%)
    nuevosItems[index].iva = nuevosItems[index].incluir_iva ? nuevosItems[index].subtotal * 0.19 : 0

    setItems(nuevosItems)
  }

  const subtotal = items.reduce((sum, item) => sum + item.subtotal, 0)
  const ivaTotal = items.reduce((sum, item) => sum + (item.incluir_iva ? item.iva : 0), 0)
  const total = subtotal + ivaTotal

  async function generarNumeroCotizacion(): Promise<string> {
    const año = new Date().getFullYear()
    const { count } = await supabase
      .from('cotizaciones')
      .select('*', { count: 'exact', head: true })
      .gte('created_at', `${año}-01-01`)
      .lt('created_at', `${año + 1}-01-01`)

    const numero = ((count || 0) + 1).toString().padStart(4, '0')
    return `COT-${año}-${numero}`
  }

  async function handleSubmit(e: React.FormEvent) {
    e.preventDefault()

    if (!formData.cliente_id) {
      alert('Debes seleccionar o crear un cliente')
      return
    }

    if (items.length === 0) {
      alert('Debes agregar al menos un producto o servicio')
      return
    }

    setLoading(true)

    try {
      const numeroCotizacion = await generarNumeroCotizacion()

      // Crear la cotización
      const cotizacionDataToInsert: any = {
        cliente_id: formData.cliente_id,
        numero_cotizacion: numeroCotizacion,
        descripcion: formData.descripcion || null,
        monto: total,
        fecha_emision: formData.fecha_emision,
        fecha_vencimiento: formData.fecha_vencimiento || null,
        estado: 'pendiente',
      }

      // Agregar campos de IVA solo si hay algún item con IVA
      if (ivaTotal > 0) {
        cotizacionDataToInsert.incluir_iva = true
        cotizacionDataToInsert.subtotal = subtotal
        cotizacionDataToInsert.iva = ivaTotal
      }

      const { data: cotizacionData, error: cotizacionError } = await supabase
        .from('cotizaciones')
        .insert([cotizacionDataToInsert])
        .select()
        .single()

      if (cotizacionError) throw cotizacionError

      // Crear los items de la cotización
      const itemsData = items.map((item, index) => ({
        cotizacion_id: cotizacionData.id,
        producto_id: item.producto_id || null,
        cantidad: item.cantidad,
        precio_unitario: item.precio_unitario,
        descripcion: item.descripcion || null,
        subtotal: item.subtotal,
        incluir_iva: item.incluir_iva,
        iva: item.incluir_iva ? item.iva : null,
        orden: index,
      }))

      const { error: itemsError } = await supabase
        .from('cotizaciones_items')
        .insert(itemsData)

      if (itemsError) throw itemsError

      alert('Cotización creada exitosamente')
      router.push(`/cotizaciones/${cotizacionData.id}`)
    } catch (error: any) {
      console.error('Error creating cotizacion:', error)
      alert(`Error al crear la cotización: ${error.message}`)
    } finally {
      setLoading(false)
    }
  }

  return (
    <div className="max-w-4xl mx-auto">
      <h1 className="text-3xl font-bold mb-6">Nueva Cotización</h1>

      <form onSubmit={handleSubmit} className="bg-white p-6 rounded-lg shadow space-y-6">
        {/* Selección/Creación de Cliente */}
        <div>
          <label className="block text-sm font-medium mb-2">Cliente *</label>
          {!creandoCliente ? (
            <div className="space-y-2">
              <select
                required={!creandoCliente}
                value={formData.cliente_id}
                onChange={(e) => setFormData({ ...formData, cliente_id: e.target.value })}
                className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
              >
                <option value="">Selecciona un cliente</option>
                {clientes.map((cliente) => (
                  <option key={cliente.id} value={cliente.id}>
                    {cliente.nombre || cliente.empresa || 'Cliente sin nombre'} {cliente.nombre && cliente.empresa ? `- ${cliente.empresa}` : ''}
                  </option>
                ))}
              </select>
              <button
                type="button"
                onClick={() => setCreandoCliente(true)}
                className="text-blue-600 hover:underline text-sm"
              >
                + Crear cliente nuevo
              </button>
            </div>
          ) : (
            <div className="border rounded p-4 space-y-3 bg-gray-50">
              <div>
                <label className="block text-sm font-medium mb-1">Nombre</label>
                <input
                  type="text"
                  value={nuevoCliente.nombre}
                  onChange={(e) => setNuevoCliente({ ...nuevoCliente, nombre: e.target.value })}
                  className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                  placeholder="Nombre completo (opcional si solo es empresa)"
                />
              </div>
              <div className="grid grid-cols-2 gap-3">
                <div>
                  <label className="block text-sm font-medium mb-1">Email</label>
                  <input
                    type="email"
                    value={nuevoCliente.email}
                    onChange={(e) => setNuevoCliente({ ...nuevoCliente, email: e.target.value })}
                    className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium mb-1">Teléfono</label>
                  <input
                    type="tel"
                    value={nuevoCliente.telefono}
                    onChange={(e) => setNuevoCliente({ ...nuevoCliente, telefono: e.target.value })}
                    className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                  />
                </div>
              </div>
              <div>
                <label className="block text-sm font-medium mb-1">Empresa</label>
                <input
                  type="text"
                  value={nuevoCliente.empresa}
                  onChange={(e) => setNuevoCliente({ ...nuevoCliente, empresa: e.target.value })}
                  className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                />
              </div>
              <div className="flex gap-2">
                <button
                  type="button"
                  onClick={crearClienteRapido}
                  className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700 transition"
                >
                  Crear Cliente
                </button>
                <button
                  type="button"
                  onClick={() => {
                    setCreandoCliente(false)
                    setNuevoCliente({ nombre: '', email: '', telefono: '', empresa: '' })
                  }}
                  className="bg-gray-300 text-gray-700 px-4 py-2 rounded hover:bg-gray-400 transition"
                >
                  Cancelar
                </button>
              </div>
            </div>
          )}
        </div>

        {/* Fechas */}
        <div className="grid grid-cols-2 gap-4">
          <div>
            <label className="block text-sm font-medium mb-1">Fecha de Emisión *</label>
            <input
              type="date"
              required
              value={formData.fecha_emision}
              onChange={(e) => setFormData({ ...formData, fecha_emision: e.target.value })}
              className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            />
          </div>
          <div>
            <label className="block text-sm font-medium mb-1">Fecha de Vencimiento</label>
            <input
              type="date"
              value={formData.fecha_vencimiento}
              onChange={(e) => setFormData({ ...formData, fecha_vencimiento: e.target.value })}
              className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            />
          </div>
        </div>

        {/* Descripción */}
        <div>
          <label className="block text-sm font-medium mb-1">Descripción General</label>
          <textarea
            value={formData.descripcion}
            onChange={(e) => setFormData({ ...formData, descripcion: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            rows={3}
            placeholder="Notas adicionales sobre la cotización"
          />
        </div>

        {/* Productos/Servicios */}
        <div>
          <div className="flex justify-between items-center mb-3">
            <label className="block text-sm font-medium">Productos y Servicios *</label>
            <button
              type="button"
              onClick={agregarItem}
              className="bg-green-600 text-white px-4 py-2 rounded hover:bg-green-700 transition text-sm"
            >
              + Agregar Producto
            </button>
          </div>

          {items.length === 0 && (
            <p className="text-gray-500 text-sm mb-4">No hay productos agregados. Haz clic en "Agregar Producto" para comenzar.</p>
          )}

          <div className="space-y-4">
            {items.map((item, index) => (
              <div key={index} className="border rounded p-4 bg-gray-50">
                <div className="grid grid-cols-12 gap-3 items-end">
                  <div className="col-span-5">
                    <label className="block text-sm font-medium mb-1">Producto/Servicio</label>
                    <select
                      value={item.producto_id}
                      onChange={(e) => actualizarItem(index, 'producto_id', e.target.value)}
                      className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                    >
                      <option value="">Selecciona un producto</option>
                      {productos.map((producto) => (
                        <option key={producto.id} value={producto.id}>
                          {producto.nombre} - ${producto.precio.toLocaleString()}
                        </option>
                      ))}
                    </select>
                  </div>
                  <div className="col-span-2">
                    <label className="block text-sm font-medium mb-1">Cantidad</label>
                    <input
                      type="number"
                      min="0.01"
                      step="0.01"
                      value={item.cantidad}
                      onChange={(e) => actualizarItem(index, 'cantidad', parseFloat(e.target.value) || 0)}
                      className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                    />
                  </div>
                  <div className="col-span-2">
                    <label className="block text-sm font-medium mb-1">Precio Unit.</label>
                    <input
                      type="number"
                      min="0"
                      step="0.01"
                      value={item.precio_unitario}
                      onChange={(e) => actualizarItem(index, 'precio_unitario', parseFloat(e.target.value) || 0)}
                      className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                    />
                  </div>
                  <div className="col-span-2">
                    <label className="block text-sm font-medium mb-1">Subtotal</label>
                    <input
                      type="text"
                      readOnly
                      value={`$${item.subtotal.toLocaleString()}`}
                      className="w-full border rounded px-3 py-2 bg-gray-100 font-semibold"
                    />
                  </div>
                  <div className="col-span-1">
                    <button
                      type="button"
                      onClick={() => eliminarItem(index)}
                      className="w-full bg-red-500 text-white px-3 py-2 rounded hover:bg-red-600 transition"
                    >
                      🗑️
                    </button>
                  </div>
                </div>
                <div className="mt-2 grid grid-cols-2 gap-3">
                  <div>
                    <label className="block text-sm font-medium mb-1">Descripción (opcional)</label>
                    <input
                      type="text"
                      value={item.descripcion}
                      onChange={(e) => actualizarItem(index, 'descripcion', e.target.value)}
                      className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                      placeholder="Descripción adicional del item"
                    />
                  </div>
                  <div className="flex items-end">
                    <div className="flex items-center gap-2 w-full">
                      <input
                        type="checkbox"
                        id={`iva_${index}`}
                        checked={item.incluir_iva}
                        onChange={(e) => actualizarItem(index, 'incluir_iva', e.target.checked)}
                        className="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                      />
                      <label htmlFor={`iva_${index}`} className="text-sm font-medium">
                        Incluir IVA (19%)
                      </label>
                      {item.incluir_iva && (
                        <span className="text-sm text-green-600 font-semibold ml-auto">
                          IVA: ${item.iva.toLocaleString()}
                        </span>
                      )}
                    </div>
                  </div>
                </div>
              </div>
            ))}
          </div>

          {items.length > 0 && (
            <div className="mt-4 p-4 bg-blue-50 rounded-lg space-y-2">
              <div className="flex justify-between items-center">
                <span className="text-sm text-gray-600">Subtotal:</span>
                <span className="text-sm font-semibold">
                  ${subtotal.toLocaleString()}
                </span>
              </div>
              {ivaTotal > 0 && (
                <div className="flex justify-between items-center">
                  <span className="text-sm text-gray-600">IVA (19%):</span>
                  <span className="text-sm font-semibold">
                    ${ivaTotal.toLocaleString()}
                  </span>
                </div>
              )}
              <div className="flex justify-between items-center border-t pt-2">
                <span className="text-lg font-semibold">Total:</span>
                <span className="text-2xl font-bold text-blue-600">
                  ${total.toLocaleString()}
                </span>
              </div>
            </div>
          )}
        </div>

        <div className="flex gap-4 pt-4">
          <button
            type="submit"
            disabled={loading || items.length === 0}
            className="bg-blue-600 text-white px-6 py-2 rounded hover:bg-blue-700 disabled:opacity-50 transition"
          >
            {loading ? 'Guardando...' : 'Crear Cotización'}
          </button>
          <button
            type="button"
            onClick={() => router.back()}
            className="bg-gray-300 text-gray-700 px-6 py-2 rounded hover:bg-gray-400 transition"
          >
            Cancelar
          </button>
        </div>
      </form>
    </div>
  )
}
