'use client'

import { useEffect, useState } from 'react'
import { supabase } from '@/lib/supabase'
import { Cotizacion, Cliente } from '@/lib/types'
import Link from 'next/link'
import { format } from 'date-fns'

export default function CotizacionesPage() {
  const [cotizaciones, setCotizaciones] = useState<(Cotizacion & { cliente?: Cliente })[]>([])
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    fetchCotizaciones()
  }, [])

  async function fetchCotizaciones() {
    try {
      const { data, error } = await supabase
        .from('cotizaciones')
        .select(`
          *,
          cliente:clientes(*)
        `)
        .order('created_at', { ascending: false })

      if (error) throw error
      setCotizaciones(data || [])
    } catch (error) {
      console.error('Error fetching cotizaciones:', error)
    } finally {
      setLoading(false)
    }
  }

  function obtenerColorEstado(estado: string): string {
    switch (estado) {
      case 'aceptada':
        return 'bg-green-100 text-green-800'
      case 'rechazada':
        return 'bg-red-100 text-red-800'
      case 'vencida':
        return 'bg-yellow-100 text-yellow-800'
      default:
        return 'bg-blue-100 text-blue-800'
    }
  }

  if (loading) {
    return (
      <div className="text-center p-8">
        <div className="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        <p className="mt-2">Cargando cotizaciones...</p>
      </div>
    )
  }

  return (
    <div>
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-3xl font-bold">Cotizaciones</h1>
        <div className="flex gap-2">
          <Link
            href="/productos"
            className="bg-gray-600 text-white px-4 py-2 rounded hover:bg-gray-700 transition"
          >
            📦 Productos
          </Link>
          <Link
            href="/cotizaciones/nuevo"
            className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700 transition"
          >
            + Nueva Cotización
          </Link>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
        {cotizaciones.map((cotizacion) => {
          const cliente = cotizacion.cliente as Cliente | undefined
          return (
            <Link
              key={cotizacion.id}
              href={`/cotizaciones/${cotizacion.id}`}
              className="bg-white p-4 rounded-lg shadow hover:shadow-lg transition"
            >
              <div className="flex justify-between items-start mb-2">
                <div>
                  <h3 className="font-bold text-lg">{cotizacion.numero_cotizacion}</h3>
                  <p className="text-sm text-gray-600">
                    {cliente?.nombre || 'Cliente desconocido'}
                  </p>
                </div>
                <span className={`px-2 py-1 rounded text-xs font-medium ${obtenerColorEstado(cotizacion.estado)}`}>
                  {cotizacion.estado}
                </span>
              </div>
              <p className="text-xl font-bold text-blue-600 mb-2">
                ${cotizacion.monto.toLocaleString()}
              </p>
              <p className="text-xs text-gray-500">
                Emisión: {format(new Date(cotizacion.fecha_emision), 'dd/MM/yyyy')}
              </p>
              {cotizacion.fecha_vencimiento && (
                <p className="text-xs text-gray-500">
                  Vence: {format(new Date(cotizacion.fecha_vencimiento), 'dd/MM/yyyy')}
                </p>
              )}
            </Link>
          )
        })}
      </div>

      {cotizaciones.length === 0 && (
        <div className="text-center p-8 bg-white rounded-lg shadow">
          <p className="text-gray-500 mb-4">No hay cotizaciones registradas.</p>
          <Link
            href="/cotizaciones/nuevo"
            className="inline-block bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700"
          >
            Crear primera cotización
          </Link>
        </div>
      )}
    </div>
  )
}
