'use client'

import { useEffect, useState } from 'react'
import { supabase } from '@/lib/supabase'
import { format, startOfMonth, endOfMonth, startOfYear, endOfYear, subMonths } from 'date-fns'
import { LineChart, Line, BarChart, Bar, PieChart, Pie, Cell, XAxis, YAxis, CartesianGrid, Tooltip, Legend, ResponsiveContainer } from 'recharts'
import jsPDF from 'jspdf'

const COLORS = ['#3B82F6', '#10B981', '#F59E0B', '#EF4444', '#8B5CF6', '#EC4899']

export default function DashboardPage() {
  const [loading, setLoading] = useState(true)
  const [periodo, setPeriodo] = useState<'mes' | 'trimestre' | 'año'>('mes')
  const [fechaInicio, setFechaInicio] = useState<string>('')
  const [fechaFin, setFechaFin] = useState<string>('')
  
  const [metricas, setMetricas] = useState({
    totalClientes: 0,
    clientesActivos: 0,
    totalSuscripciones: 0,
    suscripcionesActivas: 0,
    totalPagos: 0,
    pagosCompletados: 0,
    ingresosTotales: 0,
    ingresosEfectivo: 0,
    ingresosTransferencia: 0,
    ingresosTarjeta: 0,
  })

  const [datosMensuales, setDatosMensuales] = useState<any[]>([])
  const [datosMetodosPago, setDatosMetodosPago] = useState<any[]>([])
  const [datosEstadoPagos, setDatosEstadoPagos] = useState<any[]>([])

  useEffect(() => {
    calcularFechas()
  }, [periodo])

  useEffect(() => {
    if (fechaInicio && fechaFin) {
      fetchDatos()
    }
  }, [fechaInicio, fechaFin])

  function calcularFechas() {
    const hoy = new Date()
    let inicio: Date
    let fin: Date = hoy

    if (periodo === 'mes') {
      inicio = startOfMonth(hoy)
      fin = endOfMonth(hoy)
    } else if (periodo === 'trimestre') {
      inicio = startOfMonth(subMonths(hoy, 2))
      fin = endOfMonth(hoy)
    } else {
      inicio = startOfYear(hoy)
      fin = endOfYear(hoy)
    }

    setFechaInicio(inicio.toISOString().split('T')[0])
    setFechaFin(fin.toISOString().split('T')[0])
  }

  async function fetchDatos() {
    setLoading(true)
    try {
      // Obtener todas las métricas
      const [clientes, suscripciones, pagos] = await Promise.all([
        supabase.from('clientes').select('*'),
        supabase.from('suscripciones').select('*'),
        supabase.from('pagos').select('*')
      ])

      // Calcular métricas generales
      const totalClientes = clientes.data?.length || 0
      const clientesActivos = clientes.data?.filter(c => c.activo).length || 0
      const totalSuscripciones = suscripciones.data?.length || 0
      const suscripcionesActivas = suscripciones.data?.filter(s => s.estado === 'activa').length || 0

      // Filtrar pagos por período
      const pagosFiltrados = pagos.data?.filter(p => {
        const fechaPago = new Date(p.fecha_pago)
        return fechaPago >= new Date(fechaInicio) && fechaPago <= new Date(fechaFin)
      }) || []

      const totalPagos = pagosFiltrados.length
      const pagosCompletados = pagosFiltrados.filter(p => p.estado === 'completado').length
      const ingresosTotales = pagosFiltrados
        .filter(p => p.estado === 'completado')
        .reduce((sum, p) => sum + Number(p.monto || 0), 0)

      const ingresosEfectivo = pagosFiltrados
        .filter(p => p.estado === 'completado' && p.metodo_pago === 'efectivo')
        .reduce((sum, p) => sum + Number(p.monto || 0), 0)

      const ingresosTransferencia = pagosFiltrados
        .filter(p => p.estado === 'completado' && p.metodo_pago === 'transferencia')
        .reduce((sum, p) => sum + Number(p.monto || 0), 0)

      const ingresosTarjeta = pagosFiltrados
        .filter(p => p.estado === 'completado' && p.metodo_pago === 'tarjeta')
        .reduce((sum, p) => sum + Number(p.monto || 0), 0)

      setMetricas({
        totalClientes,
        clientesActivos,
        totalSuscripciones,
        suscripcionesActivas,
        totalPagos,
        pagosCompletados,
        ingresosTotales,
        ingresosEfectivo,
        ingresosTransferencia,
        ingresosTarjeta,
      })

      // Preparar datos para gráficos mensuales
      const meses: { [key: string]: { mes: string; ingresos: number; pagos: number } } = {}
      pagosFiltrados
        .filter(p => p.estado === 'completado')
        .forEach(p => {
          const fecha = new Date(p.fecha_pago)
          const mesKey = format(fecha, 'MMM yyyy')
          if (!meses[mesKey]) {
            meses[mesKey] = { mes: mesKey, ingresos: 0, pagos: 0 }
          }
          meses[mesKey].ingresos += Number(p.monto || 0)
          meses[mesKey].pagos += 1
        })
      setDatosMensuales(Object.values(meses))

      // Preparar datos de métodos de pago
      const metodos: { [key: string]: number } = {}
      pagosFiltrados
        .filter(p => p.estado === 'completado')
        .forEach(p => {
          const metodo = p.metodo_pago || 'otro'
          metodos[metodo] = (metodos[metodo] || 0) + Number(p.monto || 0)
        })
      setDatosMetodosPago(
        Object.entries(metodos).map(([name, value]) => ({ name, value }))
      )

      // Preparar datos de estado de pagos
      const estados: { [key: string]: number } = {}
      pagosFiltrados.forEach(p => {
        const estado = p.estado || 'pendiente'
        estados[estado] = (estados[estado] || 0) + 1
      })
      setDatosEstadoPagos(
        Object.entries(estados).map(([name, value]) => ({ name, value }))
      )
    } catch (error) {
      console.error('Error fetching datos:', error)
    } finally {
      setLoading(false)
    }
  }

  function descargarBalance() {
    const doc = new jsPDF()
    
    doc.setFontSize(20)
    doc.text('Balance General - NAVIGPS', 20, 20)
    
    doc.setFontSize(12)
    doc.text(`Período: ${periodo.charAt(0).toUpperCase() + periodo.slice(1)}`, 20, 35)
    doc.text(`Desde: ${format(new Date(fechaInicio), 'dd/MM/yyyy')}`, 20, 45)
    doc.text(`Hasta: ${format(new Date(fechaFin), 'dd/MM/yyyy')}`, 20, 55)

    let y = 70
    doc.setFontSize(14)
    doc.text('Métricas Generales', 20, y)
    y += 10

    doc.setFontSize(10)
    doc.text(`Total Clientes: ${metricas.totalClientes}`, 20, y)
    y += 7
    doc.text(`Clientes Activos: ${metricas.clientesActivos}`, 20, y)
    y += 7
    doc.text(`Total Suscripciones: ${metricas.totalSuscripciones}`, 20, y)
    y += 7
    doc.text(`Suscripciones Activas: ${metricas.suscripcionesActivas}`, 20, y)
    y += 7
    doc.text(`Total Pagos: ${metricas.totalPagos}`, 20, y)
    y += 7
    doc.text(`Pagos Completados: ${metricas.pagosCompletados}`, 20, y)
    y += 15

    doc.setFontSize(14)
    doc.text('Ingresos', 20, y)
    y += 10

    doc.setFontSize(10)
    doc.text(`Ingresos Totales: $${metricas.ingresosTotales.toLocaleString()}`, 20, y)
    y += 7
    doc.text(`Efectivo: $${metricas.ingresosEfectivo.toLocaleString()}`, 20, y)
    y += 7
    doc.text(`Transferencias: $${metricas.ingresosTransferencia.toLocaleString()}`, 20, y)
    y += 7
    doc.text(`Tarjeta: $${metricas.ingresosTarjeta.toLocaleString()}`, 20, y)

    const nombreArchivo = `balance_${periodo}_${format(new Date(), 'yyyy-MM-dd')}.pdf`
    doc.save(nombreArchivo)
  }

  if (loading) {
    return (
      <div className="text-center p-8">
        <div className="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        <p className="mt-2">Cargando dashboard...</p>
      </div>
    )
  }

  return (
    <div>
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-3xl font-bold">Dashboard</h1>
        <div className="flex gap-4 items-center">
          <select
            value={periodo}
            onChange={(e) => setPeriodo(e.target.value as 'mes' | 'trimestre' | 'año')}
            className="border rounded px-4 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="mes">Este Mes</option>
            <option value="trimestre">Este Trimestre</option>
            <option value="año">Este Año</option>
          </select>
          <button
            onClick={descargarBalance}
            className="bg-green-600 text-white px-4 py-2 rounded hover:bg-green-700 transition flex items-center gap-2"
          >
            📥 Descargar Balance
          </button>
        </div>
      </div>

      {/* Métricas principales */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
        <div className="bg-white p-6 rounded-lg shadow">
          <h3 className="text-gray-600 text-sm mb-2">Total Clientes</h3>
          <p className="text-3xl font-bold text-blue-600">{metricas.totalClientes}</p>
          <p className="text-sm text-gray-500 mt-1">{metricas.clientesActivos} activos</p>
        </div>
        <div className="bg-white p-6 rounded-lg shadow">
          <h3 className="text-gray-600 text-sm mb-2">Suscripciones</h3>
          <p className="text-3xl font-bold text-green-600">{metricas.totalSuscripciones}</p>
          <p className="text-sm text-gray-500 mt-1">{metricas.suscripcionesActivas} activas</p>
        </div>
        <div className="bg-white p-6 rounded-lg shadow">
          <h3 className="text-gray-600 text-sm mb-2">Total Pagos</h3>
          <p className="text-3xl font-bold text-purple-600">{metricas.totalPagos}</p>
          <p className="text-sm text-gray-500 mt-1">{metricas.pagosCompletados} completados</p>
        </div>
        <div className="bg-white p-6 rounded-lg shadow">
          <h3 className="text-gray-600 text-sm mb-2">Ingresos Totales</h3>
          <p className="text-3xl font-bold text-orange-600">${metricas.ingresosTotales.toLocaleString()}</p>
          <p className="text-sm text-gray-500 mt-1">Período seleccionado</p>
        </div>
      </div>

      {/* Ingresos por método de pago */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
        <div className="bg-white p-6 rounded-lg shadow">
          <h3 className="text-gray-600 text-sm mb-2">Efectivo</h3>
          <p className="text-2xl font-bold text-green-600">${metricas.ingresosEfectivo.toLocaleString()}</p>
        </div>
        <div className="bg-white p-6 rounded-lg shadow">
          <h3 className="text-gray-600 text-sm mb-2">Transferencias</h3>
          <p className="text-2xl font-bold text-blue-600">${metricas.ingresosTransferencia.toLocaleString()}</p>
        </div>
        <div className="bg-white p-6 rounded-lg shadow">
          <h3 className="text-gray-600 text-sm mb-2">Tarjeta</h3>
          <p className="text-2xl font-bold text-purple-600">${metricas.ingresosTarjeta.toLocaleString()}</p>
        </div>
      </div>

      {/* Gráficos */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-6">
        {/* Gráfico de ingresos mensuales */}
        <div className="bg-white p-6 rounded-lg shadow">
          <h3 className="text-xl font-bold mb-4">Ingresos por Mes</h3>
          <ResponsiveContainer width="100%" height={300}>
            <LineChart data={datosMensuales}>
              <CartesianGrid strokeDasharray="3 3" />
              <XAxis dataKey="mes" />
              <YAxis />
              <Tooltip formatter={(value) => `$${Number(value).toLocaleString()}`} />
              <Legend />
              <Line type="monotone" dataKey="ingresos" stroke="#3B82F6" strokeWidth={2} name="Ingresos" />
            </LineChart>
          </ResponsiveContainer>
        </div>

        {/* Gráfico de pagos por mes */}
        <div className="bg-white p-6 rounded-lg shadow">
          <h3 className="text-xl font-bold mb-4">Pagos por Mes</h3>
          <ResponsiveContainer width="100%" height={300}>
            <BarChart data={datosMensuales}>
              <CartesianGrid strokeDasharray="3 3" />
              <XAxis dataKey="mes" />
              <YAxis />
              <Tooltip />
              <Legend />
              <Bar dataKey="pagos" fill="#10B981" name="Cantidad de Pagos" />
            </BarChart>
          </ResponsiveContainer>
        </div>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Gráfico de métodos de pago */}
        <div className="bg-white p-6 rounded-lg shadow">
          <h3 className="text-xl font-bold mb-4">Ingresos por Método de Pago</h3>
          <ResponsiveContainer width="100%" height={300}>
            <PieChart>
              <Pie
                data={datosMetodosPago}
                cx="50%"
                cy="50%"
                labelLine={false}
                label={({ name, percent }) => `${name}: ${percent ? (percent * 100).toFixed(0) : 0}%`}
                outerRadius={80}
                fill="#8884d8"
                dataKey="value"
              >
                {datosMetodosPago.map((entry, index) => (
                  <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                ))}
              </Pie>
              <Tooltip formatter={(value) => `$${Number(value).toLocaleString()}`} />
            </PieChart>
          </ResponsiveContainer>
        </div>

        {/* Gráfico de estado de pagos */}
        <div className="bg-white p-6 rounded-lg shadow">
          <h3 className="text-xl font-bold mb-4">Estado de Pagos</h3>
          <ResponsiveContainer width="100%" height={300}>
            <PieChart>
              <Pie
                data={datosEstadoPagos}
                cx="50%"
                cy="50%"
                labelLine={false}
                label={({ name, percent }) => `${name}: ${percent ? (percent * 100).toFixed(0) : 0}%`}
                outerRadius={80}
                fill="#8884d8"
                dataKey="value"
              >
                {datosEstadoPagos.map((entry, index) => (
                  <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                ))}
              </Pie>
              <Tooltip />
            </PieChart>
          </ResponsiveContainer>
        </div>
      </div>
    </div>
  )
}
