'use client'

import { useState } from 'react'
import { useRouter } from 'next/navigation'
import { supabase } from '@/lib/supabase'
import { CampoFormulario, CampoFormularioTipo } from '@/lib/types'

export default function NuevoFormularioPage() {
  const router = useRouter()
  const [loading, setLoading] = useState(false)
  const [titulo, setTitulo] = useState('')
  const [descripcion, setDescripcion] = useState('')
  const [campos, setCampos] = useState<CampoFormulario[]>([])

  function agregarCampo() {
    const nuevoCampo: CampoFormulario = {
      id: Date.now().toString(),
      nombre: `campo_${campos.length + 1}`,
      etiqueta: '',
      tipo: 'text',
      requerido: false,
      orden: campos.length,
    }
    setCampos([...campos, nuevoCampo])
  }

  function eliminarCampo(id: string) {
    setCampos(campos.filter(c => c.id !== id).map((c, i) => ({ ...c, orden: i })))
  }

  function actualizarCampo(id: string, updates: Partial<CampoFormulario>) {
    setCampos(campos.map(c => c.id === id ? { ...c, ...updates } : c))
  }

  function moverCampo(id: string, direccion: 'up' | 'down') {
    const index = campos.findIndex(c => c.id === id)
    if (index === -1) return
    
    const newIndex = direccion === 'up' ? index - 1 : index + 1
    if (newIndex < 0 || newIndex >= campos.length) return

    const newCampos = [...campos]
    ;[newCampos[index], newCampos[newIndex]] = [newCampos[newIndex], newCampos[index]]
    setCampos(newCampos.map((c, i) => ({ ...c, orden: i })))
  }

  async function handleSubmit(e: React.FormEvent) {
    e.preventDefault()
    if (!titulo.trim()) {
      alert('El título es requerido')
      return
    }
    if (campos.length === 0) {
      alert('Debes agregar al menos un campo')
      return
    }

    setLoading(true)

    try {
      // Generar link público único
      const linkPublico = `f_${Date.now()}_${Math.random().toString(36).substring(7)}`

      const { error } = await supabase
        .from('formularios')
        .insert([{
          titulo,
          descripcion: descripcion || null,
          campos,
          activo: true,
          link_publico: linkPublico,
        }])

      if (error) throw error

      router.push('/formularios')
    } catch (error: any) {
      console.error('Error creating formulario:', error)
      alert(`Error al crear el formulario: ${error.message}`)
    } finally {
      setLoading(false)
    }
  }

  return (
    <div className="max-w-4xl mx-auto">
      <h1 className="text-3xl font-bold mb-6">Nuevo Formulario</h1>
      <form onSubmit={handleSubmit} className="space-y-6">
        <div className="bg-white p-6 rounded-lg shadow">
          <h2 className="text-xl font-bold mb-4">Información del Formulario</h2>
          <div className="space-y-4">
            <div>
              <label className="block text-sm font-medium mb-1">Título *</label>
              <input
                type="text"
                required
                value={titulo}
                onChange={(e) => setTitulo(e.target.value)}
                className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                placeholder="Ej: Formulario de Registro de Cliente"
              />
            </div>
            <div>
              <label className="block text-sm font-medium mb-1">Descripción</label>
              <textarea
                value={descripcion}
                onChange={(e) => setDescripcion(e.target.value)}
                className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                rows={3}
                placeholder="Descripción del formulario..."
              />
            </div>
          </div>
        </div>

        <div className="bg-white p-6 rounded-lg shadow">
          <div className="flex justify-between items-center mb-4">
            <h2 className="text-xl font-bold">Campos del Formulario</h2>
            <button
              type="button"
              onClick={agregarCampo}
              className="bg-green-600 text-white px-4 py-2 rounded hover:bg-green-700 transition"
            >
              + Agregar Campo
            </button>
          </div>

          {campos.length === 0 && (
            <p className="text-gray-500 text-center py-8">
              No hay campos. Haz clic en "Agregar Campo" para comenzar.
            </p>
          )}

          <div className="space-y-4">
            {campos.map((campo, index) => (
              <div key={campo.id} className="border rounded-lg p-4 bg-gray-50">
                <div className="flex justify-between items-start mb-3">
                  <span className="text-sm font-semibold text-gray-600">Campo {index + 1}</span>
                  <div className="flex gap-2">
                    <button
                      type="button"
                      onClick={() => moverCampo(campo.id, 'up')}
                      disabled={index === 0}
                      className="text-xs px-2 py-1 bg-gray-200 rounded disabled:opacity-50"
                    >
                      ↑
                    </button>
                    <button
                      type="button"
                      onClick={() => moverCampo(campo.id, 'down')}
                      disabled={index === campos.length - 1}
                      className="text-xs px-2 py-1 bg-gray-200 rounded disabled:opacity-50"
                    >
                      ↓
                    </button>
                    <button
                      type="button"
                      onClick={() => eliminarCampo(campo.id)}
                      className="text-xs px-2 py-1 bg-red-100 text-red-700 rounded hover:bg-red-200"
                    >
                      Eliminar
                    </button>
                  </div>
                </div>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-medium mb-1">Etiqueta *</label>
                    <input
                      type="text"
                      required
                      value={campo.etiqueta}
                      onChange={(e) => actualizarCampo(campo.id, { etiqueta: e.target.value })}
                      className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                      placeholder="Ej: Nombre completo"
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium mb-1">Tipo de Campo *</label>
                    <select
                      value={campo.tipo}
                      onChange={(e) => actualizarCampo(campo.id, { tipo: e.target.value as CampoFormularioTipo })}
                      className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                    >
                      <option value="text">Texto</option>
                      <option value="email">Email</option>
                      <option value="tel">Teléfono</option>
                      <option value="date">Fecha</option>
                      <option value="number">Número</option>
                      <option value="textarea">Área de texto</option>
                      <option value="select">Selección</option>
                      <option value="checkbox">Casilla</option>
                    </select>
                  </div>
                  <div>
                    <label className="block text-sm font-medium mb-1">Nombre del campo (ID)</label>
                    <input
                      type="text"
                      value={campo.nombre}
                      onChange={(e) => actualizarCampo(campo.id, { nombre: e.target.value })}
                      className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                      placeholder="campo_nombre"
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium mb-1">Placeholder</label>
                    <input
                      type="text"
                      value={campo.placeholder || ''}
                      onChange={(e) => actualizarCampo(campo.id, { placeholder: e.target.value })}
                      className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                      placeholder="Texto de ayuda..."
                    />
                  </div>
                  {campo.tipo === 'select' && (
                    <div className="md:col-span-2">
                      <label className="block text-sm font-medium mb-1">Opciones (una por línea)</label>
                      <textarea
                        value={campo.opciones?.join('\n') || ''}
                        onChange={(e) => actualizarCampo(campo.id, { 
                          opciones: e.target.value.split('\n').filter(o => o.trim()) 
                        })}
                        className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                        rows={3}
                        placeholder="Opción 1&#10;Opción 2&#10;Opción 3"
                      />
                    </div>
                  )}
                  <div className="flex items-center">
                    <input
                      type="checkbox"
                      checked={campo.requerido}
                      onChange={(e) => actualizarCampo(campo.id, { requerido: e.target.checked })}
                      className="mr-2"
                    />
                    <label className="text-sm font-medium">Campo requerido</label>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>

        <div className="flex gap-4">
          <button
            type="submit"
            disabled={loading}
            className="bg-blue-600 text-white px-6 py-2 rounded hover:bg-blue-700 disabled:opacity-50 transition"
          >
            {loading ? 'Guardando...' : 'Guardar Formulario'}
          </button>
          <button
            type="button"
            onClick={() => router.back()}
            className="bg-gray-300 text-gray-700 px-6 py-2 rounded hover:bg-gray-400 transition"
          >
            Cancelar
          </button>
        </div>
      </form>
    </div>
  )
}
