'use client'

import { useEffect, useState } from 'react'
import { supabase } from '@/lib/supabase'
import { Formulario } from '@/lib/types'
import Link from 'next/link'

export default function FormulariosPage() {
  const [formularios, setFormularios] = useState<Formulario[]>([])
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    fetchFormularios()
  }, [])

  async function fetchFormularios() {
    try {
      const { data, error } = await supabase
        .from('formularios')
        .select('*')
        .order('created_at', { ascending: false })

      if (error) throw error
      setFormularios(data || [])
    } catch (error) {
      console.error('Error fetching formularios:', error)
    } finally {
      setLoading(false)
    }
  }

  async function copiarLink(link: string) {
    try {
      await navigator.clipboard.writeText(link)
      alert('Link copiado al portapapeles!')
    } catch (error) {
      console.error('Error copiando link:', error)
    }
  }

  if (loading) {
    return (
      <div className="text-center p-8">
        <div className="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        <p className="mt-2">Cargando formularios...</p>
      </div>
    )
  }

  return (
    <div>
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-3xl font-bold">Formularios</h1>
        <Link
          href="/formularios/nuevo"
          className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700 transition"
        >
          + Nuevo Formulario
        </Link>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
        {formularios.map((formulario) => {
          const linkCompleto = `${window.location.origin}/formularios/publico/${formulario.link_publico}`
          return (
            <div key={formulario.id} className="bg-white p-4 rounded-lg shadow hover:shadow-lg transition">
              <div className="flex justify-between items-start mb-2">
                <h3 className="font-bold text-lg">{formulario.titulo}</h3>
                <span className={`px-2 py-1 rounded text-xs ${
                  formulario.activo ? 'bg-green-100 text-green-800' : 'bg-gray-100 text-gray-800'
                }`}>
                  {formulario.activo ? 'Activo' : 'Inactivo'}
                </span>
              </div>
              {formulario.descripcion && (
                <p className="text-sm text-gray-600 mb-3">{formulario.descripcion}</p>
              )}
              <p className="text-xs text-gray-500 mb-3">
                {formulario.campos?.length || 0} campo(s)
              </p>
              <div className="flex gap-2">
                <button
                  onClick={() => copiarLink(linkCompleto)}
                  className="flex-1 bg-blue-100 text-blue-700 px-3 py-1 rounded text-sm hover:bg-blue-200 transition"
                >
                  📋 Copiar Link
                </button>
                <Link
                  href={`/formularios/${formulario.id}`}
                  className="flex-1 bg-gray-100 text-gray-700 px-3 py-1 rounded text-sm hover:bg-gray-200 transition text-center"
                >
                  Ver
                </Link>
              </div>
            </div>
          )
        })}
      </div>

      {formularios.length === 0 && (
        <div className="text-center p-8 bg-white rounded-lg shadow">
          <p className="text-gray-500 mb-4">No hay formularios creados.</p>
          <Link
            href="/formularios/nuevo"
            className="inline-block bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700"
          >
            Crear primer formulario
          </Link>
        </div>
      )}
    </div>
  )
}
