'use client'

import { useState, useEffect } from 'react'
import { useRouter, useSearchParams } from 'next/navigation'
import { supabase } from '@/lib/supabase'
import { Cliente, Suscripcion, Producto } from '@/lib/types'

export default function NuevoPagoPage() {
  const router = useRouter()
  const searchParams = useSearchParams()
  const clienteIdParam = searchParams.get('cliente_id')
  const suscripcionIdParam = searchParams.get('suscripcion_id')
  
  const [loading, setLoading] = useState(false)
  const [clientes, setClientes] = useState<Cliente[]>([])
  const [suscripciones, setSuscripciones] = useState<Suscripcion[]>([])
  const [productos, setProductos] = useState<Producto[]>([])
  const [formData, setFormData] = useState({
    cliente_id: clienteIdParam || '',
    suscripcion_id: suscripcionIdParam || '',
    producto_id: '',
    monto: '',
    fecha_pago: new Date().toISOString().split('T')[0],
    fecha_inicio: '',
    fecha_fin: '',
    metodo_pago: '',
    referencia: '',
    estado: 'completado' as 'pendiente' | 'completado' | 'cancelado',
    notas: '',
  })

  useEffect(() => {
    fetchClientes()
    fetchProductos()
    if (clienteIdParam) {
      fetchSuscripciones(clienteIdParam)
    }
  }, [clienteIdParam])

  async function fetchClientes() {
    try {
      const { data, error } = await supabase
        .from('clientes')
        .select('*')
        .eq('activo', true)
        .order('nombre')

      if (error) throw error
      setClientes(data || [])
    } catch (error) {
      console.error('Error fetching clientes:', error)
    }
  }

  async function fetchSuscripciones(clienteId: string) {
    try {
      const { data, error } = await supabase
        .from('suscripciones')
        .select('*')
        .eq('cliente_id', clienteId)
        .eq('estado', 'activa')
        .order('created_at', { ascending: false })

      if (error) throw error
      setSuscripciones(data || [])
    } catch (error) {
      console.error('Error fetching suscripciones:', error)
    }
  }

  async function fetchProductos() {
    try {
      const { data, error } = await supabase
        .from('productos')
        .select('*')
        .eq('activo', true)
        .order('nombre')

      if (error) throw error
      setProductos(data || [])
    } catch (error) {
      console.error('Error fetching productos:', error)
    }
  }

  function handleProductoChange(productoId: string) {
    const producto = productos.find(p => p.id === productoId)
    if (producto) {
      setFormData({
        ...formData,
        producto_id: productoId,
        monto: producto.precio.toString(),
      })
    } else {
      setFormData({
        ...formData,
        producto_id: productoId,
      })
    }
  }

  function calcularFechaFin(fechaInicio: string, tipo: string): string {
    const [year, month, day] = fechaInicio.split('-').map(Number)
    let nuevoYear = year
    let nuevoMonth = month
    
    if (tipo === 'anual') {
      nuevoYear = year + 1
    } else {
      nuevoMonth = month + 1
      if (nuevoMonth > 12) {
        nuevoMonth = 1
        nuevoYear = year + 1
      }
    }
    
    return `${nuevoYear}-${nuevoMonth.toString().padStart(2, '0')}-${day.toString().padStart(2, '0')}`
  }

  async function handleClienteChange(clienteId: string) {
    setFormData({ ...formData, cliente_id: clienteId, suscripcion_id: '', fecha_inicio: '', fecha_fin: '' })
    if (clienteId) {
      await fetchSuscripciones(clienteId)
    } else {
      setSuscripciones([])
    }
  }

  function handleSuscripcionChange(suscripcionId: string) {
    const suscripcion = suscripciones.find(s => s.id === suscripcionId)
    if (suscripcion) {
      const fechaInicio = suscripcion.fecha_inicio.split('T')[0]
      const fechaFin = suscripcion.fecha_fin ? suscripcion.fecha_fin.split('T')[0] : calcularFechaFin(fechaInicio, suscripcion.tipo)
      setFormData({ 
        ...formData, 
        suscripcion_id: suscripcionId,
        monto: suscripcion.precio.toString(),
        fecha_pago: fechaInicio,
        fecha_inicio: fechaInicio,
        fecha_fin: fechaFin,
      })
    } else {
      setFormData({ ...formData, suscripcion_id: suscripcionId, fecha_inicio: '', fecha_fin: '' })
    }
  }

  async function handleSubmit(e: React.FormEvent) {
    e.preventDefault()
    setLoading(true)

    try {
      const { error } = await supabase
        .from('pagos')
        .insert([{
          cliente_id: formData.cliente_id,
          suscripcion_id: formData.suscripcion_id || null,
          producto_id: formData.producto_id || null,
          monto: parseFloat(formData.monto),
          fecha_pago: formData.fecha_pago,
          fecha_inicio: formData.fecha_inicio || formData.fecha_pago || null,
          fecha_fin: formData.fecha_fin || null,
          metodo_pago: formData.metodo_pago || null,
          referencia: formData.referencia || null,
          estado: formData.estado,
          notas: formData.notas || null,
        }])

      if (error) throw error

      if (clienteIdParam) {
        router.push(`/clientes/${clienteIdParam}`)
      } else {
        router.push('/pagos')
      }
    } catch (error: any) {
      console.error('Error creating pago:', error)
      alert(`Error al registrar el pago: ${error.message}`)
    } finally {
      setLoading(false)
    }
  }

  return (
    <div className="max-w-2xl mx-auto">
      <h1 className="text-3xl font-bold mb-6">Registrar Pago</h1>
      <form onSubmit={handleSubmit} className="bg-white p-6 rounded-lg shadow space-y-4">
        <div>
          <label className="block text-sm font-medium mb-1">Cliente *</label>
          <select
            required
            value={formData.cliente_id}
            onChange={(e) => handleClienteChange(e.target.value)}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="">Selecciona un cliente</option>
            {clientes.map((cliente) => (
              <option key={cliente.id} value={cliente.id}>
                {cliente.nombre || cliente.empresa || 'Cliente sin nombre'} {cliente.nombre && cliente.empresa ? `- ${cliente.empresa}` : ''}
              </option>
            ))}
          </select>
        </div>

        {formData.cliente_id && suscripciones.length > 0 && (
          <div>
            <label className="block text-sm font-medium mb-1">Suscripción (opcional)</label>
            <select
              value={formData.suscripcion_id}
              onChange={(e) => handleSuscripcionChange(e.target.value)}
              className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            >
              <option value="">Sin suscripción</option>
              {suscripciones.map((suscripcion) => (
                <option key={suscripcion.id} value={suscripcion.id}>
                  {suscripcion.tipo === 'anual' ? 'Anual' : 'Mensual'} - ${suscripcion.precio.toLocaleString()}
                </option>
              ))}
            </select>
          </div>
        )}

        <div>
          <label className="block text-sm font-medium mb-1">Producto o Servicio (opcional)</label>
          <select
            value={formData.producto_id}
            onChange={(e) => handleProductoChange(e.target.value)}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="">Selecciona un producto o servicio</option>
            {productos.map((producto) => (
              <option key={producto.id} value={producto.id}>
                {producto.nombre} - ${producto.precio.toLocaleString()}
              </option>
            ))}
          </select>
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Monto *</label>
          <input
            type="number"
            required
            step="0.01"
            min="0"
            value={formData.monto}
            onChange={(e) => setFormData({ ...formData, monto: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder="0.00"
          />
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Fecha de Pago *</label>
          <input
            type="date"
            required
            value={formData.fecha_pago}
            onChange={(e) => setFormData({ ...formData, fecha_pago: e.target.value, fecha_inicio: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          />
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Fecha de Inicio</label>
          <input
            type="date"
            value={formData.fecha_inicio}
            onChange={(e) => {
              const nuevaFechaInicio = e.target.value
              setFormData({ 
                ...formData, 
                fecha_inicio: nuevaFechaInicio,
                fecha_pago: nuevaFechaInicio || formData.fecha_pago
              })
            }}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          />
          <p className="text-xs text-gray-500 mt-1">Se llena automáticamente al seleccionar una suscripción</p>
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Fecha de Fin</label>
          <input
            type="date"
            value={formData.fecha_fin}
            onChange={(e) => setFormData({ ...formData, fecha_fin: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          />
          <p className="text-xs text-gray-500 mt-1">Se llena automáticamente al seleccionar una suscripción</p>
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Método de Pago</label>
          <select
            value={formData.metodo_pago}
            onChange={(e) => setFormData({ ...formData, metodo_pago: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="">Selecciona método</option>
            <option value="efectivo">Efectivo</option>
            <option value="transferencia">Transferencia</option>
            <option value="tarjeta">Tarjeta</option>
            <option value="cheque">Cheque</option>
            <option value="otro">Otro</option>
          </select>
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Referencia</label>
          <input
            type="text"
            value={formData.referencia}
            onChange={(e) => setFormData({ ...formData, referencia: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder="Número de referencia o comprobante"
          />
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Estado *</label>
          <select
            required
            value={formData.estado}
            onChange={(e) => setFormData({ ...formData, estado: e.target.value as any })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="completado">Completado</option>
            <option value="pendiente">Pendiente</option>
            <option value="cancelado">Cancelado</option>
          </select>
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Notas</label>
          <textarea
            value={formData.notas}
            onChange={(e) => setFormData({ ...formData, notas: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            rows={3}
            placeholder="Notas adicionales sobre el pago"
          />
        </div>

        <div className="flex gap-4 pt-4">
          <button
            type="submit"
            disabled={loading}
            className="bg-blue-600 text-white px-6 py-2 rounded hover:bg-blue-700 disabled:opacity-50 transition"
          >
            {loading ? 'Guardando...' : 'Registrar Pago'}
          </button>
          <button
            type="button"
            onClick={() => router.back()}
            className="bg-gray-300 text-gray-700 px-6 py-2 rounded hover:bg-gray-400 transition"
          >
            Cancelar
          </button>
        </div>
      </form>
    </div>
  )
}
