'use client'

import { useEffect, useState, useRef } from 'react'
import { useSearchParams } from 'next/navigation'
import { supabase } from '@/lib/supabase'
import { Pago, Cliente, Suscripcion } from '@/lib/types'
import Link from 'next/link'
import { format, addMonths, addYears } from 'date-fns'
import Modal from '../components/Modal'

export default function PagosPage() {
  const searchParams = useSearchParams()
  const suscripcionIdParam = searchParams.get('suscripcion_id')
  const pagoRef = useRef<{ [key: string]: HTMLTableRowElement | null }>({})
  
  const [pagos, setPagos] = useState<(Pago & { cliente?: Cliente; suscripcion?: Suscripcion })[]>([])
  const [loading, setLoading] = useState(true)
  const [filtroEstado, setFiltroEstado] = useState<string>('todos')
  const [marcandoCompletado, setMarcandoCompletado] = useState<string | null>(null)
  const [eliminandoId, setEliminandoId] = useState<string | null>(null)
  const [busqueda, setBusqueda] = useState('')
  const [modalConfirmarCompletar, setModalConfirmarCompletar] = useState<{ isOpen: boolean; pago: Pago | null; suscripcion: Suscripcion | null }>({
    isOpen: false,
    pago: null,
    suscripcion: null
  })
  const [modalConfirmarEliminar, setModalConfirmarEliminar] = useState<{ isOpen: boolean; pagoId: string | null }>({
    isOpen: false,
    pagoId: null
  })
  const [modalMensaje, setModalMensaje] = useState<{ isOpen: boolean; mensaje: string; tipo: 'success' | 'error' | 'warning' }>({
    isOpen: false,
    mensaje: '',
    tipo: 'success'
  })

  useEffect(() => {
    fetchPagos()
  }, [])

  // Scroll al pago relacionado cuando se viene desde suscripciones
  useEffect(() => {
    if (suscripcionIdParam && pagos.length > 0) {
      const pagoRelacionado = pagos.find(p => p.suscripcion_id === suscripcionIdParam)
      if (pagoRelacionado && pagoRef.current[pagoRelacionado.id]) {
        setTimeout(() => {
          pagoRef.current[pagoRelacionado.id]?.scrollIntoView({ behavior: 'smooth', block: 'center' })
          // Agregar un highlight temporal
          const row = pagoRef.current[pagoRelacionado.id]
          if (row) {
            row.classList.add('ring-4', 'ring-yellow-400', 'ring-opacity-75')
            setTimeout(() => {
              row.classList.remove('ring-4', 'ring-yellow-400', 'ring-opacity-75')
            }, 3000)
          }
        }, 500)
      }
    }
  }, [suscripcionIdParam, pagos])

  async function fetchPagos() {
    try {
      const { data, error } = await supabase
        .from('pagos')
        .select(`
          *,
          cliente:clientes(*),
          suscripcion:suscripciones(*)
        `)
        .order('created_at', { ascending: false })

      if (error) throw error
      setPagos(data || [])
    } catch (error) {
      console.error('Error fetching pagos:', error)
    } finally {
      setLoading(false)
    }
  }

  function estaVencido(pago: Pago): boolean {
    // Usar fecha_fin si existe, sino fecha_inicio, sino fecha_pago
    const fechaStr = (pago.fecha_fin || pago.fecha_inicio || pago.fecha_pago).split('T')[0]
    const hoyStr = new Date().toISOString().split('T')[0]
    return fechaStr < hoyStr
  }

  function estaPorVencer(pago: Pago): boolean {
    // Usar fecha_fin si existe, sino fecha_inicio, sino fecha_pago
    const fechaStr = (pago.fecha_fin || pago.fecha_inicio || pago.fecha_pago).split('T')[0]
    const hoyStr = new Date().toISOString().split('T')[0]
    
    // Convertir a números para calcular días
    const [year, month, day] = fechaStr.split('-').map(Number)
    const [yearHoy, monthHoy, dayHoy] = hoyStr.split('-').map(Number)
    
    const fecha = new Date(year, month - 1, day)
    const hoy = new Date(yearHoy, monthHoy - 1, dayHoy)
    
    const diasRestantes = Math.ceil((fecha.getTime() - hoy.getTime()) / (1000 * 60 * 60 * 24))
    return diasRestantes <= 7 && diasRestantes > 0 && fechaStr >= hoyStr
  }

  function obtenerColorPago(pago: Pago): string {
    if (pago.estado === 'completado') {
      return 'bg-green-50 border-green-200'
    }
    if (pago.estado === 'cancelado') {
      return 'bg-gray-50 border-gray-200'
    }
    if (estaVencido(pago)) {
      return 'bg-red-50 border-red-200 border-l-4'
    }
    if (estaPorVencer(pago)) {
      return 'bg-yellow-50 border-yellow-200 border-l-4'
    }
    return 'bg-blue-50 border-blue-200 border-l-4'
  }

  function abrirModalCompletar(pago: Pago, suscripcion: Suscripcion | null) {
    if (pago.estado === 'completado') return
    setModalConfirmarCompletar({ isOpen: true, pago, suscripcion })
  }

  async function marcarComoCompletado() {
    const pago = modalConfirmarCompletar.pago
    const suscripcion = modalConfirmarCompletar.suscripcion
    
    if (!pago || pago.estado === 'completado') return

    setMarcandoCompletado(pago.id)
    setModalConfirmarCompletar({ isOpen: false, pago: null, suscripcion: null })

    try {
      // 1. Marcar el pago actual como completado
      const { error: updateError } = await supabase
        .from('pagos')
        .update({ estado: 'completado' })
        .eq('id', pago.id)

      if (updateError) throw updateError

      // 2. Si hay una suscripción asociada, crear el siguiente pago recurrente (siempre, sin importar fecha fin)
      let siguientePagoCreado = false
      if (pago.suscripcion_id && suscripcion) {
        // Usar fecha_fin del pago actual, o calcular desde fecha_inicio, o usar fecha_pago
        const fechaBaseStr = pago.fecha_fin || pago.fecha_inicio || pago.fecha_pago
        const fechaBase = fechaBaseStr.split('T')[0]
        const [year, month, day] = fechaBase.split('-').map(Number)
        
        let siguienteYear = year
        let siguienteMonth = month
        let siguienteDay = day

        if (suscripcion.tipo === 'anual') {
          siguienteYear = year + 1
        } else {
          siguienteMonth = month + 1
          if (siguienteMonth > 12) {
            siguienteMonth = 1
            siguienteYear = year + 1
          }
        }

        const siguienteFechaInicio = `${siguienteYear}-${siguienteMonth.toString().padStart(2, '0')}-${siguienteDay.toString().padStart(2, '0')}`
        
        // Calcular fecha fin del siguiente pago
        let siguienteFechaFin: string
        if (suscripcion.tipo === 'anual') {
          siguienteFechaFin = `${siguienteYear + 1}-${siguienteMonth.toString().padStart(2, '0')}-${siguienteDay.toString().padStart(2, '0')}`
        } else {
          let finMonth = siguienteMonth + 1
          let finYear = siguienteYear
          if (finMonth > 12) {
            finMonth = 1
            finYear = siguienteYear + 1
          }
          siguienteFechaFin = `${finYear}-${finMonth.toString().padStart(2, '0')}-${siguienteDay.toString().padStart(2, '0')}`
        }

        // Crear siempre el siguiente pago recurrente (sin verificar fecha fin de suscripción)
        const { data: nuevoPagoData, error: nuevoPagoError } = await supabase
          .from('pagos')
          .insert([{
            cliente_id: pago.cliente_id,
            suscripcion_id: pago.suscripcion_id,
            monto: suscripcion.precio,
            fecha_pago: siguienteFechaInicio,
            fecha_inicio: siguienteFechaInicio,
            fecha_fin: siguienteFechaFin,
            estado: 'pendiente',
            notas: `Pago ${suscripcion.tipo === 'anual' ? 'anual' : 'mensual'} recurrente - Generado automáticamente`,
          }])
          .select()

        if (nuevoPagoError) {
          console.error('Error creating siguiente pago:', nuevoPagoError)
          setModalMensaje({
            isOpen: true,
            mensaje: `Error al crear el siguiente pago recurrente: ${nuevoPagoError.message}`,
            tipo: 'error'
          })
        } else if (nuevoPagoData && nuevoPagoData.length > 0) {
          siguientePagoCreado = true
          console.log('✅ Siguiente pago creado exitosamente:', nuevoPagoData[0])
        }
      }

      // Recargar los pagos
      await fetchPagos()
      
      if (siguientePagoCreado) {
        setModalMensaje({
          isOpen: true,
          mensaje: '✅ Pago marcado como completado. Se creó el siguiente pago recurrente automáticamente.',
          tipo: 'success'
        })
      } else if (pago.suscripcion_id && suscripcion) {
        setModalMensaje({
          isOpen: true,
          mensaje: '⚠️ Pago marcado como completado, pero no se pudo crear el siguiente pago recurrente. Verifica la consola para más detalles.',
          tipo: 'warning'
        })
      } else {
        setModalMensaje({
          isOpen: true,
          mensaje: '✅ Pago marcado como completado exitosamente.',
          tipo: 'success'
        })
      }
    } catch (error: any) {
      console.error('Error updating pago:', error)
      setModalMensaje({
        isOpen: true,
        mensaje: `Error al marcar el pago: ${error.message}`,
        tipo: 'error'
      })
    } finally {
      setMarcandoCompletado(null)
    }
  }

  function abrirModalEliminar(id: string) {
    setModalConfirmarEliminar({ isOpen: true, pagoId: id })
  }

  async function eliminarPago() {
    if (!modalConfirmarEliminar.pagoId) return

    setEliminandoId(modalConfirmarEliminar.pagoId)
    setModalConfirmarEliminar({ isOpen: false, pagoId: null })

    try {
      const { error } = await supabase
        .from('pagos')
        .delete()
        .eq('id', modalConfirmarEliminar.pagoId)

      if (error) throw error

      setModalMensaje({
        isOpen: true,
        mensaje: 'Pago eliminado exitosamente',
        tipo: 'success'
      })
      fetchPagos()
    } catch (error: any) {
      console.error('Error deleting pago:', error)
      setModalMensaje({
        isOpen: true,
        mensaje: `Error al eliminar el pago: ${error.message}`,
        tipo: 'error'
      })
    } finally {
      setEliminandoId(null)
    }
  }

  const pagosFiltrados = (filtroEstado === 'todos' 
    ? pagos 
    : pagos.filter(p => p.estado === filtroEstado)
  ).filter(p => {
    if (!busqueda) return true
    const termino = busqueda.toLowerCase()
    const cliente = p.cliente as Cliente | undefined
    return (
      cliente?.nombre?.toLowerCase().includes(termino) ||
      cliente?.empresa?.toLowerCase().includes(termino) ||
      p.monto.toString().includes(termino) ||
      p.metodo_pago?.toLowerCase().includes(termino) ||
      p.referencia?.toLowerCase().includes(termino) ||
      p.estado?.toLowerCase().includes(termino)
    )
  })

  if (loading) {
    return (
      <div className="text-center p-8">
        <div className="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        <p className="mt-2">Cargando pagos...</p>
      </div>
    )
  }

  return (
    <div>
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-3xl font-bold">Pagos</h1>
        <Link
          href="/pagos/nuevo"
          className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700 transition"
        >
          + Registrar Pago
        </Link>
      </div>

      <div className="mb-4">
        <input
          type="text"
          placeholder="🔍 Buscar por cliente, monto, método de pago o referencia..."
          value={busqueda}
          onChange={(e) => setBusqueda(e.target.value)}
          className="w-full max-w-md border rounded-lg px-4 py-2 mb-4 focus:outline-none focus:ring-2 focus:ring-blue-500"
        />
        <div className="flex gap-2">
          <button
            onClick={() => setFiltroEstado('todos')}
            className={`px-4 py-2 rounded ${filtroEstado === 'todos' ? 'bg-blue-600 text-white' : 'bg-gray-200'}`}
          >
            Todos
          </button>
        <button
          onClick={() => setFiltroEstado('pendiente')}
          className={`px-4 py-2 rounded ${filtroEstado === 'pendiente' ? 'bg-blue-600 text-white' : 'bg-gray-200'}`}
        >
          Pendientes
        </button>
        <button
          onClick={() => setFiltroEstado('completado')}
          className={`px-4 py-2 rounded ${filtroEstado === 'completado' ? 'bg-blue-600 text-white' : 'bg-gray-200'}`}
        >
          Completados
        </button>
        </div>
      </div>

      {pagosFiltrados.length === 0 && pagos.length > 0 && (
        <div className="text-center p-8 bg-white rounded-lg shadow mb-4">
          <p className="text-gray-500">No se encontraron pagos que coincidan con "{busqueda}"</p>
        </div>
      )}

      <div className="overflow-x-auto">
        <table className="w-full bg-white rounded-lg shadow">
          <thead className="bg-gray-50">
            <tr>
              <th className="px-4 py-3 text-left text-sm font-semibold">Cliente</th>
              <th className="px-4 py-3 text-left text-sm font-semibold">Monto</th>
              <th className="px-4 py-3 text-left text-sm font-semibold">Fecha Inicio</th>
              <th className="px-4 py-3 text-left text-sm font-semibold">Fecha Fin</th>
              <th className="px-4 py-3 text-left text-sm font-semibold">Método</th>
              <th className="px-4 py-3 text-left text-sm font-semibold">Estado</th>
              <th className="px-4 py-3 text-center text-sm font-semibold">✓</th>
              <th className="px-4 py-3 text-left text-sm font-semibold">Acciones</th>
            </tr>
          </thead>
          <tbody>
            {pagosFiltrados.map((pago) => {
              const cliente = pago.cliente as Cliente | undefined
              const suscripcion = pago.suscripcion as Suscripcion | null
              const vencido = estaVencido(pago)
              const porVencer = estaPorVencer(pago)
              const colorFila = obtenerColorPago(pago)
              
              let estadoTexto: string = pago.estado
              let estadoColor = 'bg-gray-100 text-gray-800'
              
              if (pago.estado === 'completado') {
                estadoTexto = 'Completado'
                estadoColor = 'bg-green-100 text-green-800'
              } else if (pago.estado === 'cancelado') {
                estadoTexto = 'Cancelado'
                estadoColor = 'bg-gray-100 text-gray-800'
              } else if (vencido) {
                estadoTexto = 'Vencido'
                estadoColor = 'bg-red-100 text-red-800'
              } else if (porVencer) {
                estadoTexto = 'Por vencer'
                estadoColor = 'bg-yellow-100 text-yellow-800'
              } else {
                estadoTexto = 'En curso'
                estadoColor = 'bg-blue-100 text-blue-800'
              }
              
              return (
                <tr 
                  key={pago.id} 
                  ref={(el) => { if (el) pagoRef.current[pago.id] = el }}
                  className={`${colorFila} border-t hover:opacity-80 transition`}
                >
                  <td className="px-4 py-3">
                    <Link href={`/clientes/${pago.cliente_id}`} className="text-blue-600 hover:underline font-medium">
                      {cliente?.nombre || cliente?.empresa || 'Cliente desconocido'}
                    </Link>
                  </td>
                  <td className="px-4 py-3 font-semibold">${pago.monto.toLocaleString()}</td>
                  <td className="px-4 py-3">
                    {(() => {
                      const fecha = (pago.fecha_inicio || pago.fecha_pago).split('T')[0]
                      const [year, month, day] = fecha.split('-')
                      return `${day}/${month}/${year}`
                    })()}
                  </td>
                  <td className="px-4 py-3">
                    {pago.fecha_fin ? (() => {
                      const fecha = pago.fecha_fin.split('T')[0]
                      const [year, month, day] = fecha.split('-')
                      return `${day}/${month}/${year}`
                    })() : '-'}
                  </td>
                  <td className="px-4 py-3">{pago.metodo_pago || '-'}</td>
                  <td className="px-4 py-3">
                    <span className={`px-2 py-1 rounded text-xs font-medium ${estadoColor}`}>
                      {estadoTexto}
                    </span>
                  </td>
                  <td className="px-4 py-3 text-center">
                    {pago.estado !== 'completado' && (
                      <button
                        onClick={() => abrirModalCompletar(pago, suscripcion)}
                        disabled={marcandoCompletado === pago.id}
                        className={`w-6 h-6 border-2 rounded flex items-center justify-center transition mx-auto ${
                          marcandoCompletado === pago.id 
                            ? 'border-gray-400 bg-gray-200 cursor-not-allowed' 
                            : 'border-gray-400 hover:border-green-500 hover:bg-green-50 cursor-pointer'
                        }`}
                        title="Marcar como completado"
                      >
                        {marcandoCompletado === pago.id ? (
                          <span className="text-xs">...</span>
                        ) : (
                          <span className="text-green-600 text-sm">✓</span>
                        )}
                      </button>
                    )}
                    {pago.estado === 'completado' && (
                      <span className="text-green-600 text-xl" title="Pago completado">✓</span>
                    )}
                  </td>
                  <td className="px-4 py-3">
                    <div className="flex gap-2 items-center">
                      <Link
                        href={`/pagos/${pago.id}`}
                        className="text-blue-600 hover:underline text-sm font-medium"
                      >
                        Ver
                      </Link>
                      <Link
                        href={`/pagos/${pago.id}/editar`}
                        className="text-green-600 hover:underline text-sm font-medium"
                      >
                        Editar
                      </Link>
                      <Link
                        href={`/pagos/${pago.id}/factura`}
                        className="text-purple-600 hover:underline text-sm font-medium"
                      >
                        FAC
                      </Link>
                      <button
                        onClick={() => abrirModalEliminar(pago.id)}
                        disabled={eliminandoId === pago.id}
                        className="text-red-600 hover:text-red-800 disabled:opacity-50 transition text-sm font-medium"
                        title="Eliminar pago"
                      >
                        {eliminandoId === pago.id ? '...' : '🗑️'}
                      </button>
                    </div>
                  </td>
                </tr>
              )
            })}
          </tbody>
        </table>
      </div>

      {pagosFiltrados.length === 0 && (
        <div className="text-center p-8 bg-white rounded-lg shadow">
          <p className="text-gray-500 mb-4">No hay pagos registrados.</p>
          <Link
            href="/pagos/nuevo"
            className="inline-block bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700"
          >
            Registrar primer pago
          </Link>
        </div>
      )}

      {/* Modal de confirmación para completar pago */}
      <Modal
        isOpen={modalConfirmarCompletar.isOpen}
        onClose={() => setModalConfirmarCompletar({ isOpen: false, pago: null, suscripcion: null })}
        title="Completar Pago"
        message="¿Marcar este pago como completado? Esto creará automáticamente el siguiente pago pendiente si hay una suscripción asociada."
        onConfirm={marcarComoCompletado}
        confirmText="Completar"
        cancelText="Cancelar"
        type="info"
      />

      {/* Modal de confirmación para eliminar */}
      <Modal
        isOpen={modalConfirmarEliminar.isOpen}
        onClose={() => setModalConfirmarEliminar({ isOpen: false, pagoId: null })}
        title="Eliminar Pago"
        message="¿Estás seguro de eliminar este pago? Esta acción no se puede deshacer."
        onConfirm={eliminarPago}
        confirmText="Eliminar"
        cancelText="Cancelar"
        type="warning"
      />

      {/* Modal de mensaje */}
      <Modal
        isOpen={modalMensaje.isOpen}
        onClose={() => setModalMensaje({ isOpen: false, mensaje: '', tipo: 'success' })}
        title={modalMensaje.tipo === 'success' ? 'Éxito' : modalMensaje.tipo === 'error' ? 'Error' : 'Advertencia'}
        message={modalMensaje.mensaje}
        type={modalMensaje.tipo}
      />
    </div>
  )
}
