'use client'

import { useEffect, useState } from 'react'
import { supabase } from '@/lib/supabase'
import { Producto } from '@/lib/types'
import Link from 'next/link'
import Modal from '../components/Modal'

export default function ProductosPage() {
  const [productos, setProductos] = useState<Producto[]>([])
  const [loading, setLoading] = useState(true)
  const [mostrandoFormulario, setMostrandoFormulario] = useState(false)
  const [editandoId, setEditandoId] = useState<string | null>(null)
  const [eliminandoId, setEliminandoId] = useState<string | null>(null)
  const [formData, setFormData] = useState({
    nombre: '',
    descripcion: '',
    precio: '',
  })
  const [modalConfirmar, setModalConfirmar] = useState<{ isOpen: boolean; productoId: string | null }>({
    isOpen: false,
    productoId: null
  })
  const [modalMensaje, setModalMensaje] = useState<{ isOpen: boolean; mensaje: string; tipo: 'success' | 'error' }>({
    isOpen: false,
    mensaje: '',
    tipo: 'success'
  })

  useEffect(() => {
    fetchProductos()
  }, [])

  async function fetchProductos() {
    try {
      const { data, error } = await supabase
        .from('productos')
        .select('*')
        .order('nombre')

      if (error) throw error
      setProductos(data || [])
    } catch (error) {
      console.error('Error fetching productos:', error)
    } finally {
      setLoading(false)
    }
  }

  async function handleSubmit(e: React.FormEvent) {
    e.preventDefault()

    try {
      const productoData = {
        nombre: formData.nombre,
        descripcion: formData.descripcion || null,
        precio: parseFloat(formData.precio),
      }

      if (editandoId) {
        const { error } = await supabase
          .from('productos')
          .update(productoData)
          .eq('id', editandoId)

        if (error) throw error
        setModalMensaje({
          isOpen: true,
          mensaje: 'Producto actualizado exitosamente',
          tipo: 'success'
        })
      } else {
        const { error } = await supabase
          .from('productos')
          .insert([productoData])

        if (error) throw error
        setModalMensaje({
          isOpen: true,
          mensaje: 'Producto creado exitosamente',
          tipo: 'success'
        })
      }

      setFormData({ nombre: '', descripcion: '', precio: '' })
      setMostrandoFormulario(false)
      setEditandoId(null)
      fetchProductos()
    } catch (error: any) {
      console.error('Error saving producto:', error)
      setModalMensaje({
        isOpen: true,
        mensaje: `Error: ${error.message}`,
        tipo: 'error'
      })
    }
  }

  function iniciarEdicion(producto: Producto) {
    setFormData({
      nombre: producto.nombre,
      descripcion: producto.descripcion || '',
      precio: producto.precio.toString(),
    })
    setEditandoId(producto.id)
    setMostrandoFormulario(true)
  }

  function abrirModalEliminar(id: string) {
    setModalConfirmar({ isOpen: true, productoId: id })
  }

  async function eliminarProducto() {
    if (!modalConfirmar.productoId) return

    setEliminandoId(modalConfirmar.productoId)
    setModalConfirmar({ isOpen: false, productoId: null })

    try {
      const { error } = await supabase
        .from('productos')
        .delete()
        .eq('id', modalConfirmar.productoId)

      if (error) throw error

      setModalMensaje({
        isOpen: true,
        mensaje: 'Producto eliminado exitosamente',
        tipo: 'success'
      })
      fetchProductos()
    } catch (error: any) {
      console.error('Error deleting producto:', error)
      setModalMensaje({
        isOpen: true,
        mensaje: `Error: ${error.message}`,
        tipo: 'error'
      })
    } finally {
      setEliminandoId(null)
    }
  }

  async function toggleActivo(id: string, activo: boolean) {
    try {
      const { error } = await supabase
        .from('productos')
        .update({ activo: !activo })
        .eq('id', id)

      if (error) throw error
      fetchProductos()
    } catch (error: any) {
      console.error('Error updating producto:', error)
      alert(`Error: ${error.message}`)
    }
  }

  if (loading) {
    return (
      <div className="text-center p-8">
        <div className="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        <p className="mt-2">Cargando productos...</p>
      </div>
    )
  }

  return (
    <div>
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-3xl font-bold">Productos y Servicios</h1>
        <button
          onClick={() => {
            setFormData({ nombre: '', descripcion: '', precio: '' })
            setEditandoId(null)
            setMostrandoFormulario(true)
          }}
          className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700 transition"
        >
          + Nuevo Producto
        </button>
      </div>

      {mostrandoFormulario && (
        <div className="bg-white p-6 rounded-lg shadow mb-6">
          <h2 className="text-2xl font-bold mb-4">
            {editandoId ? 'Editar Producto' : 'Nuevo Producto'}
          </h2>
          <form onSubmit={handleSubmit} className="space-y-4">
            <div>
              <label className="block text-sm font-medium mb-1">Nombre *</label>
              <input
                type="text"
                required
                value={formData.nombre}
                onChange={(e) => setFormData({ ...formData, nombre: e.target.value })}
                className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                placeholder="Ej: Servicio de GPS Mensual"
              />
            </div>
            <div>
              <label className="block text-sm font-medium mb-1">Descripción</label>
              <textarea
                value={formData.descripcion}
                onChange={(e) => setFormData({ ...formData, descripcion: e.target.value })}
                className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                rows={3}
                placeholder="Descripción detallada del producto o servicio"
              />
            </div>
            <div>
              <label className="block text-sm font-medium mb-1">Precio *</label>
              <input
                type="number"
                required
                step="0.01"
                min="0"
                value={formData.precio}
                onChange={(e) => setFormData({ ...formData, precio: e.target.value })}
                className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                placeholder="0.00"
              />
            </div>
            <div className="flex gap-4">
              <button
                type="submit"
                className="bg-blue-600 text-white px-6 py-2 rounded hover:bg-blue-700 transition"
              >
                {editandoId ? 'Actualizar' : 'Crear'}
              </button>
              <button
                type="button"
                onClick={() => {
                  setMostrandoFormulario(false)
                  setEditandoId(null)
                  setFormData({ nombre: '', descripcion: '', precio: '' })
                }}
                className="bg-gray-300 text-gray-700 px-6 py-2 rounded hover:bg-gray-400 transition"
              >
                Cancelar
              </button>
            </div>
          </form>
        </div>
      )}

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
        {productos.map((producto) => (
          <div
            key={producto.id}
            className={`bg-white p-4 rounded-lg shadow hover:shadow-lg transition ${
              !producto.activo ? 'opacity-60' : ''
            }`}
          >
            <div className="flex justify-between items-start mb-2">
              <h3 className="font-bold text-lg">{producto.nombre}</h3>
              <button
                onClick={() => toggleActivo(producto.id, producto.activo)}
                className={`px-2 py-1 rounded text-xs ${
                  producto.activo
                    ? 'bg-green-100 text-green-800'
                    : 'bg-gray-100 text-gray-800'
                }`}
              >
                {producto.activo ? 'Activo' : 'Inactivo'}
              </button>
            </div>
            {producto.descripcion && (
              <p className="text-sm text-gray-600 mb-2">{producto.descripcion}</p>
            )}
            <p className="text-xl font-bold text-blue-600 mb-3">
              ${producto.precio.toLocaleString()}
            </p>
            <div className="flex gap-2">
              <button
                onClick={() => iniciarEdicion(producto)}
                className="flex-1 bg-blue-500 text-white px-3 py-1 rounded text-sm hover:bg-blue-600 transition"
              >
                ✏️ Editar
              </button>
              <button
                onClick={() => abrirModalEliminar(producto.id)}
                disabled={eliminandoId === producto.id}
                className="flex-1 bg-red-500 text-white px-3 py-1 rounded text-sm hover:bg-red-600 disabled:opacity-50 transition"
              >
                {eliminandoId === producto.id ? '...' : '🗑️ Eliminar'}
              </button>
            </div>
          </div>
        ))}
      </div>

      {productos.length === 0 && (
        <div className="text-center p-8 bg-white rounded-lg shadow">
          <p className="text-gray-500 mb-4">No hay productos registrados.</p>
          <button
            onClick={() => setMostrandoFormulario(true)}
            className="inline-block bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700"
          >
            Crear primer producto
          </button>
        </div>
      )}

      {/* Modal de confirmación para eliminar */}
      <Modal
        isOpen={modalConfirmar.isOpen}
        onClose={() => setModalConfirmar({ isOpen: false, productoId: null })}
        title="Eliminar Producto"
        message="¿Estás seguro de eliminar este producto?"
        onConfirm={eliminarProducto}
        confirmText="Eliminar"
        cancelText="Cancelar"
        type="warning"
      />

      {/* Modal de mensaje */}
      <Modal
        isOpen={modalMensaje.isOpen}
        onClose={() => setModalMensaje({ isOpen: false, mensaje: '', tipo: 'success' })}
        title={modalMensaje.tipo === 'success' ? 'Éxito' : 'Error'}
        message={modalMensaje.mensaje}
        type={modalMensaje.tipo}
      />
    </div>
  )
}
