'use client'

import { useState, useEffect } from 'react'
import { useRouter, useSearchParams } from 'next/navigation'
import { supabase } from '@/lib/supabase'
import { Cliente } from '@/lib/types'

export default function NuevoRecordatorioPage() {
  const router = useRouter()
  const searchParams = useSearchParams()
  const clienteIdParam = searchParams.get('cliente_id')
  
  const [loading, setLoading] = useState(false)
  const [clientes, setClientes] = useState<Cliente[]>([])
  const [formData, setFormData] = useState({
    cliente_id: clienteIdParam || '',
    titulo: '',
    descripcion: '',
    fecha_recordatorio: new Date().toISOString().slice(0, 16),
    tipo: '',
  })

  useEffect(() => {
    fetchClientes()
  }, [])

  async function fetchClientes() {
    try {
      const { data, error } = await supabase
        .from('clientes')
        .select('*')
        .eq('activo', true)
        .order('nombre')

      if (error) throw error
      setClientes(data || [])
    } catch (error) {
      console.error('Error fetching clientes:', error)
    }
  }

  async function handleSubmit(e: React.FormEvent) {
    e.preventDefault()
    setLoading(true)

    try {
      const { error } = await supabase
        .from('recordatorios')
        .insert([{
          cliente_id: formData.cliente_id,
          titulo: formData.titulo,
          descripcion: formData.descripcion || null,
          fecha_recordatorio: formData.fecha_recordatorio,
          tipo: formData.tipo || null,
          completado: false,
        }])

      if (error) throw error

      if (clienteIdParam) {
        router.push(`/clientes/${clienteIdParam}`)
      } else {
        router.push('/recordatorios')
      }
    } catch (error: any) {
      console.error('Error creating recordatorio:', error)
      alert(`Error al crear el recordatorio: ${error.message}`)
    } finally {
      setLoading(false)
    }
  }

  return (
    <div className="max-w-2xl mx-auto">
      <h1 className="text-3xl font-bold mb-6">Nuevo Recordatorio</h1>
      <form onSubmit={handleSubmit} className="bg-white p-6 rounded-lg shadow space-y-4">
        <div>
          <label className="block text-sm font-medium mb-1">Cliente *</label>
          <select
            required
            value={formData.cliente_id}
            onChange={(e) => setFormData({ ...formData, cliente_id: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="">Selecciona un cliente</option>
            {clientes.map((cliente) => (
              <option key={cliente.id} value={cliente.id}>
                {cliente.nombre || cliente.empresa || 'Cliente sin nombre'} {cliente.nombre && cliente.empresa ? `- ${cliente.empresa}` : ''}
              </option>
            ))}
          </select>
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Título *</label>
          <input
            type="text"
            required
            value={formData.titulo}
            onChange={(e) => setFormData({ ...formData, titulo: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder="Ej: Llamar para seguimiento"
          />
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Descripción</label>
          <textarea
            value={formData.descripcion}
            onChange={(e) => setFormData({ ...formData, descripcion: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            rows={3}
            placeholder="Detalles del recordatorio..."
          />
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Fecha y Hora *</label>
          <input
            type="datetime-local"
            required
            value={formData.fecha_recordatorio}
            onChange={(e) => setFormData({ ...formData, fecha_recordatorio: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          />
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Tipo</label>
          <select
            value={formData.tipo}
            onChange={(e) => setFormData({ ...formData, tipo: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="">Sin tipo</option>
            <option value="Llamada">Llamada</option>
            <option value="Reunión">Reunión</option>
            <option value="Pago">Pago</option>
            <option value="Seguimiento">Seguimiento</option>
            <option value="Renovación">Renovación</option>
            <option value="Otro">Otro</option>
          </select>
        </div>

        <div className="flex gap-4 pt-4">
          <button
            type="submit"
            disabled={loading}
            className="bg-blue-600 text-white px-6 py-2 rounded hover:bg-blue-700 disabled:opacity-50 transition"
          >
            {loading ? 'Guardando...' : 'Guardar Recordatorio'}
          </button>
          <button
            type="button"
            onClick={() => router.back()}
            className="bg-gray-300 text-gray-700 px-6 py-2 rounded hover:bg-gray-400 transition"
          >
            Cancelar
          </button>
        </div>
      </form>
    </div>
  )
}
