'use client'

import { useEffect, useState } from 'react'
import { supabase } from '@/lib/supabase'
import { Recordatorio, Cliente } from '@/lib/types'
import Link from 'next/link'
import { format } from 'date-fns'

export default function RecordatoriosPage() {
  const [recordatorios, setRecordatorios] = useState<(Recordatorio & { cliente?: Cliente })[]>([])
  const [loading, setLoading] = useState(true)
  const [filtro, setFiltro] = useState<string>('todos')

  useEffect(() => {
    fetchRecordatorios()
  }, [])

  async function fetchRecordatorios() {
    try {
      const { data, error } = await supabase
        .from('recordatorios')
        .select(`
          *,
          cliente:clientes(*)
        `)
        .order('fecha_recordatorio', { ascending: true })

      if (error) throw error
      setRecordatorios(data || [])
    } catch (error) {
      console.error('Error fetching recordatorios:', error)
    } finally {
      setLoading(false)
    }
  }

  async function toggleCompletado(id: string, completado: boolean) {
    try {
      const { error } = await supabase
        .from('recordatorios')
        .update({ completado: !completado })
        .eq('id', id)

      if (error) throw error
      fetchRecordatorios()
    } catch (error) {
      console.error('Error updating recordatorio:', error)
    }
  }

  const recordatoriosFiltrados = filtro === 'todos' 
    ? recordatorios 
    : filtro === 'pendientes'
    ? recordatorios.filter(r => !r.completado)
    : recordatorios.filter(r => r.completado)

  const recordatoriosProximos = recordatorios
    .filter(r => !r.completado && new Date(r.fecha_recordatorio) >= new Date())
    .slice(0, 5)

  if (loading) {
    return (
      <div className="text-center p-8">
        <div className="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        <p className="mt-2">Cargando recordatorios...</p>
      </div>
    )
  }

  return (
    <div>
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-3xl font-bold">Recordatorios</h1>
        <Link
          href="/recordatorios/nuevo"
          className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700 transition"
        >
          + Nuevo Recordatorio
        </Link>
      </div>

      {recordatoriosProximos.length > 0 && (
        <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4 mb-6">
          <h2 className="font-bold mb-2">Próximos Recordatorios</h2>
          <div className="space-y-2">
            {recordatoriosProximos.map((recordatorio) => {
              const cliente = recordatorio.cliente as Cliente | undefined
              return (
                <div key={recordatorio.id} className="flex justify-between items-center">
                  <div>
                    <p className="font-medium">{recordatorio.titulo}</p>
                    <p className="text-sm text-gray-600">
                      {cliente?.nombre} - {format(new Date(recordatorio.fecha_recordatorio), 'dd/MM/yyyy HH:mm')}
                    </p>
                  </div>
                </div>
              )
            })}
          </div>
        </div>
      )}

      <div className="mb-4 flex gap-2">
        <button
          onClick={() => setFiltro('todos')}
          className={`px-4 py-2 rounded ${filtro === 'todos' ? 'bg-blue-600 text-white' : 'bg-gray-200'}`}
        >
          Todos
        </button>
        <button
          onClick={() => setFiltro('pendientes')}
          className={`px-4 py-2 rounded ${filtro === 'pendientes' ? 'bg-blue-600 text-white' : 'bg-gray-200'}`}
        >
          Pendientes
        </button>
        <button
          onClick={() => setFiltro('completados')}
          className={`px-4 py-2 rounded ${filtro === 'completados' ? 'bg-blue-600 text-white' : 'bg-gray-200'}`}
        >
          Completados
        </button>
      </div>

      <div className="space-y-4">
        {recordatoriosFiltrados.map((recordatorio) => {
          const cliente = recordatorio.cliente as Cliente | undefined
          const fechaRecordatorio = new Date(recordatorio.fecha_recordatorio)
          const esPasado = fechaRecordatorio < new Date()
          
          return (
            <div
              key={recordatorio.id}
              className={`bg-white p-4 rounded-lg shadow ${
                recordatorio.completado ? 'opacity-60' : ''
              } ${esPasado && !recordatorio.completado ? 'border-l-4 border-red-500' : ''}`}
            >
              <div className="flex justify-between items-start">
                <div className="flex-1">
                  <div className="flex items-center gap-2 mb-2">
                    <input
                      type="checkbox"
                      checked={recordatorio.completado}
                      onChange={() => toggleCompletado(recordatorio.id, recordatorio.completado)}
                      className="mr-2"
                    />
                    <h3 className={`font-bold text-lg ${recordatorio.completado ? 'line-through' : ''}`}>
                      {recordatorio.titulo}
                    </h3>
                    {recordatorio.tipo && (
                      <span className="px-2 py-1 bg-blue-100 text-blue-800 rounded text-xs">
                        {recordatorio.tipo}
                      </span>
                    )}
                  </div>
                  {recordatorio.descripcion && (
                    <p className="text-gray-600 mb-2">{recordatorio.descripcion}</p>
                  )}
                  <div className="flex gap-4 text-sm text-gray-500">
                    <Link href={`/clientes/${recordatorio.cliente_id}`} className="text-blue-600 hover:underline">
                      👤 {cliente?.nombre || 'Cliente desconocido'}
                    </Link>
                    <span>
                      📅 {format(fechaRecordatorio, 'dd/MM/yyyy HH:mm')}
                    </span>
                  </div>
                </div>
              </div>
            </div>
          )
        })}
      </div>

      {recordatoriosFiltrados.length === 0 && (
        <div className="text-center p-8 bg-white rounded-lg shadow">
          <p className="text-gray-500 mb-4">No hay recordatorios.</p>
          <Link
            href="/recordatorios/nuevo"
            className="inline-block bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700"
          >
            Crear primer recordatorio
          </Link>
        </div>
      )}
    </div>
  )
}
