'use client'

import { useState, useEffect } from 'react'
import { useRouter, useSearchParams } from 'next/navigation'
import { supabase } from '@/lib/supabase'
import { Cliente, Producto } from '@/lib/types'

export default function NuevaSuscripcionPage() {
  const router = useRouter()
  const searchParams = useSearchParams()
  const clienteIdParam = searchParams.get('cliente_id')
  
  const [loading, setLoading] = useState(false)
  const [clientes, setClientes] = useState<Cliente[]>([])
  const [productos, setProductos] = useState<Producto[]>([])
  const [busquedaCliente, setBusquedaCliente] = useState('')
  const fechaInicioInicial = new Date().toISOString().split('T')[0]
  const [formData, setFormData] = useState({
    cliente_id: clienteIdParam || '',
    tipo: 'mensual' as 'mensual' | 'anual',
    precio: '',
    fecha_inicio: fechaInicioInicial,
    fecha_fin: '',
    patente: '',
    producto_id: '',
    cantidad: '1',
    incluir_iva: false,
  })
  const [usarFechaFinManual, setUsarFechaFinManual] = useState(false)

  // Calcular fecha_fin inicial
  useEffect(() => {
    if (!usarFechaFinManual && formData.fecha_inicio) {
      const fechaCalculada = calcularFechaFin(formData.fecha_inicio, formData.tipo)
      if (formData.fecha_fin !== fechaCalculada) {
        setFormData(prev => ({ ...prev, fecha_fin: fechaCalculada }))
      }
    }
  }, [formData.fecha_inicio, formData.tipo, usarFechaFinManual])

  useEffect(() => {
    fetchClientes()
    fetchProductos()
  }, [])

  async function fetchClientes() {
    try {
      const { data, error } = await supabase
        .from('clientes')
        .select('*')
        .eq('activo', true)
        .order('nombre')

      if (error) throw error
      setClientes(data || [])
    } catch (error) {
      console.error('Error fetching clientes:', error)
    }
  }

  async function fetchProductos() {
    try {
      const { data, error } = await supabase
        .from('productos')
        .select('*')
        .order('nombre')

      if (error) throw error
      setProductos(data || [])
      console.log('Productos cargados:', data?.length || 0)
    } catch (error) {
      console.error('Error fetching productos:', error)
    }
  }

  function handleProductoChange(productoId: string) {
    const producto = productos.find(p => p.id === productoId)
    const cantidad = parseFloat(formData.cantidad) || 1
    if (producto) {
      const precioTotal = producto.precio * cantidad
      setFormData({
        ...formData,
        producto_id: productoId,
        precio: precioTotal.toString()
      })
    } else {
      setFormData({
        ...formData,
        producto_id: '',
        precio: ''
      })
    }
  }

  function handleCantidadChange(cantidad: string) {
    const cantidadNum = parseFloat(cantidad) || 1
    setFormData({
      ...formData,
      cantidad: cantidad
    })
    
    // Si hay un producto seleccionado, recalcular el precio
    if (formData.producto_id) {
      const producto = productos.find(p => p.id === formData.producto_id)
      if (producto) {
        const precioTotal = producto.precio * cantidadNum
        setFormData(prev => ({
          ...prev,
          cantidad: cantidad,
          precio: precioTotal.toString()
        }))
      }
    }
  }

  function calcularFechaFin(fechaInicio: string, tipo: string): string {
    // Validar que la fecha tenga el formato correcto
    if (!fechaInicio || typeof fechaInicio !== 'string') {
      return ''
    }
    
    // Trabajar directamente con strings YYYY-MM-DD para evitar problemas de zona horaria
    const partes = fechaInicio.split('-')
    if (partes.length !== 3) {
      return ''
    }
    
    const year = parseInt(partes[0], 10)
    const month = parseInt(partes[1], 10)
    const day = parseInt(partes[2], 10)
    
    // Validar que los valores sean números válidos
    if (isNaN(year) || isNaN(month) || isNaN(day)) {
      return ''
    }
    
    let nuevoYear = year
    let nuevoMonth = month
    let nuevoDay = day
    
    if (tipo === 'anual') {
      nuevoYear = year + 1
    } else {
      nuevoMonth = month + 1
      // Ajustar si el mes pasa de 12
      if (nuevoMonth > 12) {
        nuevoMonth = 1
        nuevoYear = year + 1
      }
    }
    
    // Formatear con ceros a la izquierda
    const mesStr = nuevoMonth.toString().padStart(2, '0')
    const diaStr = nuevoDay.toString().padStart(2, '0')
    
    return `${nuevoYear}-${mesStr}-${diaStr}`
  }

  async function handleSubmit(e: React.FormEvent) {
    e.preventDefault()
    setLoading(true)

    try {
      // Usar fecha_fin manual si fue ingresada, sino calcular automáticamente
      const fechaFin = usarFechaFinManual && formData.fecha_fin 
        ? formData.fecha_fin 
        : calcularFechaFin(formData.fecha_inicio, formData.tipo)
      const precio = parseFloat(formData.precio)

      // 1. Crear la suscripción
      const cantidadNum = parseFloat(formData.cantidad) || 1
      const { data: suscripcionData, error: suscripcionError } = await supabase
        .from('suscripciones')
        .insert([{
          cliente_id: formData.cliente_id,
          tipo: formData.tipo,
          precio: precio, // Precio total (precio unitario × cantidad)
          fecha_inicio: formData.fecha_inicio,
          fecha_fin: fechaFin,
          patente: formData.patente || null,
          producto_id: formData.producto_id || null,
          cantidad: cantidadNum, // Guardar la cantidad
          incluir_iva: formData.incluir_iva,
          estado: 'activa',
        }])
        .select()
        .single()

      if (suscripcionError) throw suscripcionError

      // 2. Crear automáticamente el primer pago pendiente
      if (suscripcionData) {
        // Usar siempre la fecha de inicio de la suscripción para el pago
        const fechaInicioPago = formData.fecha_inicio // Ya está en formato YYYY-MM-DD
        const fechaFinPago = fechaFin // Ya calculada anteriormente

        // El monto del pago debe ser el NETO (precio × cantidad), sin IVA
        // El IVA se calculará en la factura si incluir_iva está activo
        const montoPago = precio // Este es el precio neto (ya incluye cantidad si se calculó)

        // Crear el objeto del pago
        const pagoData: any = {
          cliente_id: formData.cliente_id,
          suscripcion_id: suscripcionData.id,
          monto: montoPago, // Monto NETO (sin IVA)
          fecha_pago: fechaInicioPago,
          estado: 'pendiente',
          notas: `Pago inicial de suscripción ${formData.tipo}`,
          incluir_iva: formData.incluir_iva,
          producto_id: formData.producto_id || null,
        }

        // Agregar fecha_inicio y fecha_fin si los campos existen en la tabla
        // (si no existen, Supabase los ignorará o dará error, lo manejaremos)
        pagoData.fecha_inicio = fechaInicioPago
        pagoData.fecha_fin = fechaFinPago

        const { data: pagoDataResult, error: pagoError } = await supabase
          .from('pagos')
          .insert([pagoData])
          .select()

        if (pagoError) {
          console.error('Error creating pago:', pagoError)
          
          // Si el error es porque los campos fecha_inicio/fecha_fin no existen,
          // intentar crear el pago sin esos campos
          if (pagoError.message.includes('column') && (pagoError.message.includes('fecha_inicio') || pagoError.message.includes('fecha_fin'))) {
            console.log('Los campos fecha_inicio/fecha_fin no existen, creando pago sin ellos...')
            const { error: pagoError2 } = await supabase
              .from('pagos')
              .insert([{
                cliente_id: formData.cliente_id,
                suscripcion_id: suscripcionData.id,
                monto: precio,
                fecha_pago: fechaInicioPago,
                estado: 'pendiente',
                notas: `Pago inicial de suscripción ${formData.tipo}`,
              }])

            if (pagoError2) {
              console.error('Error creating pago (sin fechas):', pagoError2)
              alert(`⚠️ Advertencia: La suscripción se creó pero no se pudo crear el pago automático.\n\nError: ${pagoError2.message}\n\nPor favor, crea el pago manualmente.`)
            } else {
              console.log('Pago creado exitosamente (sin fecha_inicio/fecha_fin)')
            }
          } else {
            alert(`⚠️ Advertencia: La suscripción se creó pero no se pudo crear el pago automático.\n\nError: ${pagoError.message}\n\nPor favor, crea el pago manualmente.`)
          }
        } else {
          console.log('✅ Pago creado exitosamente:', pagoDataResult)
        }
      }

      if (clienteIdParam) {
        router.push(`/clientes/${clienteIdParam}`)
      } else {
        router.push('/suscripciones')
      }
    } catch (error: any) {
      console.error('Error creating suscripcion:', error)
      alert(`Error al crear la suscripción: ${error.message}`)
    } finally {
      setLoading(false)
    }
  }

  return (
    <div className="max-w-2xl mx-auto">
      <h1 className="text-3xl font-bold mb-6">Nueva Suscripción</h1>
      <form onSubmit={handleSubmit} className="bg-white p-6 rounded-lg shadow space-y-4">
        <div>
          <label className="block text-sm font-medium mb-1">Cliente *</label>
          <input
            type="text"
            placeholder="🔍 Buscar cliente por nombre, empresa, email o teléfono..."
            value={busquedaCliente}
            onChange={(e) => setBusquedaCliente(e.target.value)}
            className="w-full border rounded px-3 py-2 mb-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          />
          <select
            required
            value={formData.cliente_id}
            onChange={(e) => {
              setFormData({ ...formData, cliente_id: e.target.value })
              // Limpiar búsqueda al seleccionar
              setBusquedaCliente('')
            }}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="">Selecciona un cliente</option>
            {(() => {
              const clientesFiltrados = clientes.filter((cliente) => {
                if (!busquedaCliente) return true
                const termino = busquedaCliente.toLowerCase()
                return (
                  cliente.nombre?.toLowerCase().includes(termino) ||
                  cliente.empresa?.toLowerCase().includes(termino) ||
                  cliente.email?.toLowerCase().includes(termino) ||
                  cliente.telefono?.toLowerCase().includes(termino) ||
                  cliente.rfc?.toLowerCase().includes(termino)
                )
              })
              return clientesFiltrados.map((cliente) => (
                <option key={cliente.id} value={cliente.id}>
                  {cliente.nombre || cliente.empresa || 'Cliente sin nombre'} {cliente.nombre && cliente.empresa ? `- ${cliente.empresa}` : ''}
                </option>
              ))
            })()}
          </select>
          {busquedaCliente && (() => {
            const clientesFiltrados = clientes.filter((cliente) => {
              const termino = busquedaCliente.toLowerCase()
              return (
                cliente.nombre?.toLowerCase().includes(termino) ||
                cliente.empresa?.toLowerCase().includes(termino) ||
                cliente.email?.toLowerCase().includes(termino) ||
                cliente.telefono?.toLowerCase().includes(termino) ||
                cliente.rfc?.toLowerCase().includes(termino)
              )
            })
            return (
              <p className="text-xs text-gray-500 mt-1">
                {clientesFiltrados.length === 0 
                  ? 'No se encontraron clientes'
                  : `${clientesFiltrados.length} cliente(s) encontrado(s)`
                }
              </p>
            )
          })()}
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Tipo de Suscripción *</label>
          <select
            required
            value={formData.tipo}
            onChange={(e) => {
              const nuevoTipo = e.target.value as 'mensual' | 'anual'
              setFormData({ 
                ...formData, 
                tipo: nuevoTipo,
                fecha_fin: usarFechaFinManual ? formData.fecha_fin : (formData.fecha_inicio ? calcularFechaFin(formData.fecha_inicio, nuevoTipo) : '')
              })
            }}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="mensual">Mensual</option>
            <option value="anual">Anual</option>
          </select>
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Producto o Servicio</label>
          <select
            value={formData.producto_id}
            onChange={(e) => handleProductoChange(e.target.value)}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="">Selecciona un producto o servicio (opcional)</option>
            {productos.length === 0 ? (
              <option value="" disabled>No hay productos disponibles. Crea productos en la sección "Productos"</option>
            ) : (
              productos.map((producto) => (
                <option key={producto.id} value={producto.id}>
                  {producto.nombre} - ${producto.precio.toLocaleString()} {!producto.activo ? '(Inactivo)' : ''}
                </option>
              ))
            )}
          </select>
          <p className="text-xs text-gray-500 mt-1">
            {productos.length === 0 
              ? 'No hay productos creados. Ve a "Productos" para crear uno.'
              : `Al seleccionar un producto, el precio se rellenará automáticamente. ${productos.length} producto(s) disponible(s).`
            }
          </p>
        </div>

        {formData.producto_id && (
          <div>
            <label className="block text-sm font-medium mb-1">Cantidad *</label>
            <input
              type="number"
              required
              min="1"
              step="1"
              value={formData.cantidad}
              onChange={(e) => handleCantidadChange(e.target.value)}
              className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            />
            <p className="text-xs text-gray-500 mt-1">El precio se calculará automáticamente (precio unitario × cantidad)</p>
          </div>
        )}

        <div>
          <label className="block text-sm font-medium mb-1">Precio *</label>
          <input
            type="number"
            required
            step="0.01"
            min="0"
            value={formData.precio}
            onChange={(e) => setFormData({ ...formData, precio: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder="0.00"
          />
        </div>

        <div className="flex items-center">
          <input
            type="checkbox"
            id="incluir_iva"
            checked={formData.incluir_iva}
            onChange={(e) => setFormData({ ...formData, incluir_iva: e.target.checked })}
            className="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
          />
          <label htmlFor="incluir_iva" className="ml-2 text-sm font-medium text-gray-700">
            Incluir IVA (19%)
          </label>
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Patente / Dispositivo</label>
          <input
            type="text"
            value={formData.patente}
            onChange={(e) => setFormData({ ...formData, patente: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder="Ej: ABC123, GPS-001, etc."
          />
          <p className="text-xs text-gray-500 mt-1">Identificador del dispositivo GPS asociado a esta suscripción</p>
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Fecha de Inicio *</label>
          <input
            type="date"
            required
            value={formData.fecha_inicio}
            onChange={(e) => {
              const nuevaFechaInicio = e.target.value
              setFormData({ 
                ...formData, 
                fecha_inicio: nuevaFechaInicio,
                fecha_fin: usarFechaFinManual ? formData.fecha_fin : (nuevaFechaInicio ? calcularFechaFin(nuevaFechaInicio, formData.tipo) : '')
              })
            }}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          />
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Fecha de Fin</label>
          <input
            type="date"
            value={formData.fecha_fin || (formData.fecha_inicio ? calcularFechaFin(formData.fecha_inicio, formData.tipo) : '')}
            onChange={(e) => {
              setFormData({ ...formData, fecha_fin: e.target.value })
              setUsarFechaFinManual(true)
            }}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          />
          {formData.fecha_inicio && (
            <div className="mt-2 flex items-center gap-2">
              <p className="text-sm text-gray-600">
                <strong>Calculada automáticamente:</strong>{' '}
                {formData.fecha_inicio ? calcularFechaFin(formData.fecha_inicio, formData.tipo) : 'N/A'}
                {' '}({formData.tipo === 'anual' ? '1 año' : '1 mes'} después del inicio)
              </p>
              {usarFechaFinManual && formData.fecha_fin && (
                <button
                  type="button"
                  onClick={() => {
                    setFormData({ 
                      ...formData, 
                      fecha_fin: formData.fecha_inicio ? calcularFechaFin(formData.fecha_inicio, formData.tipo) : '' 
                    })
                    setUsarFechaFinManual(false)
                  }}
                  className="text-sm text-blue-600 hover:underline"
                >
                  Usar fecha calculada
                </button>
              )}
            </div>
          )}
        </div>

        <div className="flex gap-4 pt-4">
          <button
            type="submit"
            disabled={loading}
            className="bg-blue-600 text-white px-6 py-2 rounded hover:bg-blue-700 disabled:opacity-50 transition"
          >
            {loading ? 'Guardando...' : 'Guardar Suscripción'}
          </button>
          <button
            type="button"
            onClick={() => router.back()}
            className="bg-gray-300 text-gray-700 px-6 py-2 rounded hover:bg-gray-400 transition"
          >
            Cancelar
          </button>
        </div>
      </form>
    </div>
  )
}
