'use client'

import { useEffect, useState } from 'react'
import { supabase } from '@/lib/supabase'
import { Suscripcion, Cliente } from '@/lib/types'
import Link from 'next/link'
import { format } from 'date-fns'
import Modal from '../components/Modal'

export default function SuscripcionesPage() {
  const [suscripciones, setSuscripciones] = useState<(Suscripcion & { cliente?: Cliente })[]>([])
  const [loading, setLoading] = useState(true)
  const [filtroEstado, setFiltroEstado] = useState<string>('todas')
  const [eliminandoId, setEliminandoId] = useState<string | null>(null)
  const [busqueda, setBusqueda] = useState('')
  const [modalConfirmar, setModalConfirmar] = useState<{ isOpen: boolean; suscripcionId: string | null }>({
    isOpen: false,
    suscripcionId: null
  })
  const [modalMensaje, setModalMensaje] = useState<{ isOpen: boolean; mensaje: string; tipo: 'success' | 'error' }>({
    isOpen: false,
    mensaje: '',
    tipo: 'success'
  })

  useEffect(() => {
    fetchSuscripciones()
  }, [])

  async function fetchSuscripciones() {
    try {
      const { data, error } = await supabase
        .from('suscripciones')
        .select(`
          *,
          cliente:clientes(*)
        `)
        .order('created_at', { ascending: false })

      if (error) throw error
      setSuscripciones(data || [])
    } catch (error) {
      console.error('Error fetching suscripciones:', error)
    } finally {
      setLoading(false)
    }
  }

  function abrirModalEliminar(id: string) {
    setModalConfirmar({ isOpen: true, suscripcionId: id })
  }

  async function eliminarSuscripcion() {
    if (!modalConfirmar.suscripcionId) return

    setEliminandoId(modalConfirmar.suscripcionId)
    setModalConfirmar({ isOpen: false, suscripcionId: null })

    try {
      const { error } = await supabase
        .from('suscripciones')
        .delete()
        .eq('id', modalConfirmar.suscripcionId)

      if (error) throw error

      setModalMensaje({
        isOpen: true,
        mensaje: 'Suscripción eliminada exitosamente',
        tipo: 'success'
      })
      fetchSuscripciones()
    } catch (error: any) {
      console.error('Error deleting suscripcion:', error)
      setModalMensaje({
        isOpen: true,
        mensaje: `Error al eliminar la suscripción: ${error.message}`,
        tipo: 'error'
      })
    } finally {
      setEliminandoId(null)
    }
  }

  function calcularFechaFin(fechaInicio: string, tipo: string): string {
    const inicio = new Date(fechaInicio)
    const fin = new Date(inicio)
    
    if (tipo === 'anual') {
      fin.setFullYear(fin.getFullYear() + 1)
    } else {
      fin.setMonth(fin.getMonth() + 1)
    }
    
    return fin.toISOString().split('T')[0]
  }

  function estaVencida(fechaFin?: string): boolean {
    if (!fechaFin) return false
    return new Date(fechaFin) < new Date()
  }

  function estaPorVencer(fechaFin?: string): boolean {
    if (!fechaFin) return false
    const fin = new Date(fechaFin)
    const hoy = new Date()
    const diasRestantes = Math.ceil((fin.getTime() - hoy.getTime()) / (1000 * 60 * 60 * 24))
    return diasRestantes <= 7 && diasRestantes > 0 // Por vencer en 7 días o menos
  }

  function obtenerColorSuscripcion(suscripcion: Suscripcion): string {
    if (suscripcion.estado === 'cancelada') {
      return 'bg-gray-100 border-gray-300'
    }
    if (estaVencida(suscripcion.fecha_fin || undefined)) {
      return 'bg-red-50 border-red-300 border-l-4'
    }
    if (estaPorVencer(suscripcion.fecha_fin || undefined)) {
      return 'bg-yellow-50 border-yellow-300 border-l-4'
    }
    if (suscripcion.estado === 'activa') {
      return 'bg-green-50 border-green-300 border-l-4'
    }
    return 'bg-white border-gray-200'
  }

  const suscripcionesFiltradas = (filtroEstado === 'todas' 
    ? suscripciones 
    : suscripciones.filter(s => {
        if (filtroEstado === 'vencidas') {
          return estaVencida(s.fecha_fin || undefined)
        }
        return s.estado === filtroEstado
      })
  ).filter(s => {
    if (!busqueda) return true
    const termino = busqueda.toLowerCase()
    const cliente = s.cliente as Cliente | undefined
    return (
      cliente?.nombre?.toLowerCase().includes(termino) ||
      cliente?.empresa?.toLowerCase().includes(termino) ||
      s.tipo?.toLowerCase().includes(termino) ||
      s.precio.toString().includes(termino) ||
      s.estado?.toLowerCase().includes(termino)
    )
  })

  if (loading) {
    return (
      <div className="text-center p-8">
        <div className="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        <p className="mt-2">Cargando suscripciones...</p>
      </div>
    )
  }

  return (
    <div>
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-3xl font-bold">Suscripciones</h1>
        <Link
          href="/suscripciones/nuevo"
          className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700 transition"
        >
          + Nueva Suscripción
        </Link>
      </div>

      <div className="mb-4">
        <input
          type="text"
          placeholder="🔍 Buscar por cliente, tipo, precio o estado..."
          value={busqueda}
          onChange={(e) => setBusqueda(e.target.value)}
          className="w-full max-w-md border rounded-lg px-4 py-2 mb-4 focus:outline-none focus:ring-2 focus:ring-blue-500"
        />
        <div className="flex gap-2">
          <button
            onClick={() => setFiltroEstado('todas')}
            className={`px-4 py-2 rounded ${filtroEstado === 'todas' ? 'bg-blue-600 text-white' : 'bg-gray-200'}`}
          >
            Todas
          </button>
        <button
          onClick={() => setFiltroEstado('activa')}
          className={`px-4 py-2 rounded ${filtroEstado === 'activa' ? 'bg-blue-600 text-white' : 'bg-gray-200'}`}
        >
          Activas
        </button>
        <button
          onClick={() => setFiltroEstado('vencida')}
          className={`px-4 py-2 rounded ${filtroEstado === 'vencida' ? 'bg-blue-600 text-white' : 'bg-gray-200'}`}
        >
          Vencidas
        </button>
        <button
          onClick={() => setFiltroEstado('cancelada')}
          className={`px-4 py-2 rounded ${filtroEstado === 'cancelada' ? 'bg-blue-600 text-white' : 'bg-gray-200'}`}
        >
          Canceladas
        </button>
        </div>
      </div>

      {suscripcionesFiltradas.length === 0 && suscripciones.length > 0 && (
        <div className="text-center p-8 bg-white rounded-lg shadow mb-4">
          <p className="text-gray-500">No se encontraron suscripciones que coincidan con "{busqueda}"</p>
        </div>
      )}

      <div className="bg-white rounded-lg shadow overflow-hidden">
        <div className="overflow-x-auto">
          <table className="w-full">
            <thead className="bg-gray-50 border-b border-gray-200">
              <tr>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Cliente</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Patente</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Tipo</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Precio</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Inicio</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Vence</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Estado</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Acciones</th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {suscripcionesFiltradas.map((suscripcion) => {
                const cliente = suscripcion.cliente as Cliente | undefined
                const vencida = estaVencida(suscripcion.fecha_fin || undefined)
                const porVencer = estaPorVencer(suscripcion.fecha_fin || undefined)
                
                  let estadoTexto: string = suscripcion.estado
                let estadoColor = 'bg-gray-100 text-gray-800'
                let filaColor = ''
                
                if (vencida) {
                  estadoTexto = 'Vencida'
                  estadoColor = 'bg-red-100 text-red-800'
                  filaColor = 'bg-red-50'
                } else if (porVencer) {
                  estadoTexto = 'Por vencer'
                  estadoColor = 'bg-yellow-100 text-yellow-800'
                  filaColor = 'bg-yellow-50'
                } else if (suscripcion.estado === 'activa') {
                  estadoTexto = 'En curso'
                  estadoColor = 'bg-green-100 text-green-800'
                  filaColor = 'bg-green-50'
                }
                
                const fechaInicio = suscripcion.fecha_inicio.split('T')[0]
                const [yearInicio, monthInicio, dayInicio] = fechaInicio.split('-')
                const fechaInicioFormateada = `${dayInicio}/${monthInicio}/${yearInicio}`
                
                const fechaFin = suscripcion.fecha_fin ? suscripcion.fecha_fin.split('T')[0] : null
                const fechaFinFormateada = fechaFin ? (() => {
                  const [yearFin, monthFin, dayFin] = fechaFin.split('-')
                  return `${dayFin}/${monthFin}/${yearFin}`
                })() : 'N/A'
                
                return (
                  <tr key={suscripcion.id} className={`hover:bg-gray-50 ${filaColor}`}>
                    <td className="px-4 py-3 whitespace-nowrap">
                      <Link href={`/suscripciones/${suscripcion.id}`} className="text-sm font-medium text-gray-900 hover:text-blue-600">
                        {cliente?.nombre || cliente?.empresa || 'Cliente sin nombre'}
                      </Link>
                    </td>
                    <td className="px-4 py-3 whitespace-nowrap text-sm text-gray-500">
                      {suscripcion.patente ? `🚗 ${suscripcion.patente}` : '-'}
                    </td>
                    <td className="px-4 py-3 whitespace-nowrap text-sm text-gray-500">
                      {suscripcion.tipo === 'anual' ? '📅 Anual' : '📆 Mensual'}
                    </td>
                    <td className="px-4 py-3 whitespace-nowrap text-sm font-semibold text-gray-900">
                      ${suscripcion.precio.toLocaleString()}
                    </td>
                    <td className="px-4 py-3 whitespace-nowrap text-sm text-gray-500">
                      {fechaInicioFormateada}
                    </td>
                    <td className="px-4 py-3 whitespace-nowrap text-sm text-gray-500">
                      {fechaFinFormateada}
                    </td>
                    <td className="px-4 py-3 whitespace-nowrap">
                      <span className={`px-2 py-1 rounded text-xs font-medium ${estadoColor}`}>
                        {estadoTexto}
                      </span>
                    </td>
                    <td className="px-4 py-3 whitespace-nowrap text-sm font-medium">
                      <div className="flex gap-2">
                        <Link
                          href={`/suscripciones/${suscripcion.id}/editar`}
                          className="bg-blue-500 text-white px-2 py-1 rounded hover:bg-blue-600 transition text-xs"
                          title="Editar"
                        >
                          ✏️
                        </Link>
                        <Link
                          href={`/pagos?suscripcion_id=${suscripcion.id}`}
                          className="bg-green-500 text-white px-2 py-1 rounded hover:bg-green-600 transition text-xs"
                          title="Ver Pago"
                        >
                          💰
                        </Link>
                        <button
                          onClick={() => abrirModalEliminar(suscripcion.id)}
                          disabled={eliminandoId === suscripcion.id}
                          className="bg-red-500 text-white px-2 py-1 rounded hover:bg-red-600 disabled:opacity-50 transition text-xs"
                          title="Eliminar"
                        >
                          {eliminandoId === suscripcion.id ? '...' : '🗑️'}
                        </button>
                      </div>
                    </td>
                  </tr>
                )
              })}
            </tbody>
          </table>
        </div>
      </div>

      {suscripcionesFiltradas.length === 0 && (
        <div className="text-center p-8 bg-white rounded-lg shadow">
          <p className="text-gray-500 mb-4">No hay suscripciones.</p>
          <Link
            href="/suscripciones/nuevo"
            className="inline-block bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700"
          >
            Crear primera suscripción
          </Link>
        </div>
      )}

      {/* Modal de confirmación para eliminar */}
      <Modal
        isOpen={modalConfirmar.isOpen}
        onClose={() => setModalConfirmar({ isOpen: false, suscripcionId: null })}
        title="Eliminar Suscripción"
        message="¿Estás seguro de eliminar esta suscripción? Esta acción no se puede deshacer."
        onConfirm={eliminarSuscripcion}
        confirmText="Eliminar"
        cancelText="Cancelar"
        type="warning"
      />

      {/* Modal de mensaje */}
      <Modal
        isOpen={modalMensaje.isOpen}
        onClose={() => setModalMensaje({ isOpen: false, mensaje: '', tipo: 'success' })}
        title={modalMensaje.tipo === 'success' ? 'Éxito' : 'Error'}
        message={modalMensaje.mensaje}
        type={modalMensaje.tipo}
      />
    </div>
  )
}
