// API helper para consultar información de empresas desde ruts.info
// Documentación: https://ruts.info/

export interface RutsInfoActivity {
  activity_code: string
  activity_description: string
  activity_date: string
  iva_affects: boolean
  tax_category: string
}

export interface RutsInfoAddress {
  validity: boolean
  address_date: string
  address_type: string
  street: string
  street_number: string
  block: string | null
  apartment: string | null
  neighborhood: string | null
  city: string
  district: string
  region: string
}

export interface RutsInfoCompany {
  code: number
  verification_digit: string
  business_name: string
  valid_from_date: string
  valid_until_date: string | null
  activities: RutsInfoActivity[]
  addresses: RutsInfoAddress[]
}

/**
 * Consulta información de una empresa desde ruts.info
 * @param rut RUT de la empresa sin puntos ni guión (ej: "770738512")
 * @param apiKey API Key de ruts.info
 * @returns Información de la empresa o null si hay error
 */
export async function consultarEmpresaRutsInfo(
  rut: string
): Promise<RutsInfoCompany | null> {
  try {
    // Limpiar el RUT (remover puntos, espacios y guión)
    const rutLimpio = rut.replace(/\./g, '').replace(/\s/g, '').replace(/-/g, '').trim()
    
    // Validar formato básico de RUT chileno (7-9 dígitos)
    if (!/^\d{7,9}$/.test(rutLimpio)) {
      console.error('Formato de RUT inválido:', rut)
      return null
    }

    // Construir la URL con query params
    const url = new URL('/api/ruts-info', window.location.origin)
    url.searchParams.set('rut', rutLimpio)
    
    // Usar la API route de Next.js como proxy (evita problemas de CORS)
    const response = await fetch(url.toString(), {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
      },
    })

    if (!response.ok) {
      const errorData = await response.json().catch(() => ({}))
      console.error('Error en respuesta de ruts.info:', response.status, errorData)
      return null
    }

    const data: RutsInfoCompany = await response.json()

    return data
  } catch (error) {
    console.error('Error consultando ruts.info:', error)
    return null
  }
}

/**
 * Formatea la dirección completa desde los datos de ruts.info
 */
export function formatearDireccionRutsInfo(data: RutsInfoCompany): string {
  if (!data.addresses || data.addresses.length === 0) {
    return ''
  }

  // Usar la primera dirección válida
  const direccion = data.addresses.find(addr => addr.validity) || data.addresses[0]
  
  const partes: string[] = []
  
  if (direccion.street) {
    const calle = `${direccion.street} ${direccion.street_number || ''}`.trim()
    if (calle) partes.push(calle)
  }
  
  if (direccion.apartment) partes.push(`Depto ${direccion.apartment}`)
  if (direccion.district) partes.push(direccion.district)
  if (direccion.city) partes.push(direccion.city)
  if (direccion.region) partes.push(direccion.region)
  
  return partes.join(', ')
}

/**
 * Obtiene el email DTE desde las actividades (si está disponible)
 */
export function obtenerEmailDTE(data: RutsInfoCompany): string | null {
  // ruts.info no proporciona email directamente
  // Esto se puede completar manualmente o desde otra fuente
  return null
}

/**
 * Obtiene el giro principal de la empresa
 */
export function obtenerGiroPrincipal(data: RutsInfoCompany): string | null {
  if (!data.activities || data.activities.length === 0) {
    return null
  }
  
  // Retornar la primera actividad como giro principal
  return data.activities[0].activity_description || null
}
