// API helper para consultar información de empresas desde simpleapi.cl
// Documentación: https://documentacion.simpleapi.cl/
// Plan gratuito: 10 consultas al mes

export interface SimpleApiCompany {
  rut?: string
  razon_social?: string
  nombre_fantasia?: string
  giro?: string
  direccion?: string
  comuna?: string
  ciudad?: string
  region?: string
  telefono?: string
  email?: string
  sitio_web?: string
  fecha_inicio?: string
  estado?: string
}

/**
 * Consulta información de una empresa desde simpleapi.cl
 * @param rut RUT de la empresa con o sin formato (ej: "77073851-2" o "770738512")
 * @returns Información de la empresa o null si hay error
 */
export async function consultarEmpresaSimpleApi(
  rut: string
): Promise<SimpleApiCompany | null> {
  try {
    // Limpiar el RUT (remover puntos, espacios y guión)
    const rutLimpio = rut.replace(/\./g, '').replace(/\s/g, '').replace(/-/g, '').trim()
    
    // Validar formato básico de RUT chileno (7-9 dígitos)
    if (!/^\d{7,9}$/.test(rutLimpio)) {
      console.error('Formato de RUT inválido:', rut)
      return null
    }

    // Construir la URL con query params
    const url = new URL('/api/simpleapi', window.location.origin)
    url.searchParams.set('rut', rutLimpio)
    
    // Usar la API route de Next.js como proxy (evita problemas de CORS)
    const response = await fetch(url.toString(), {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
      },
    })

    if (!response.ok) {
      const errorData = await response.json().catch(() => ({}))
      console.error('Error en respuesta de simpleapi.cl:', response.status, errorData)
      return null
    }

    const data: SimpleApiCompany = await response.json()

    return data
  } catch (error) {
    console.error('Error consultando simpleapi.cl:', error)
    return null
  }
}

/**
 * Formatea la dirección completa desde los datos de simpleapi.cl
 */
export function formatearDireccionSimpleApi(data: SimpleApiCompany): string {
  const partes: string[] = []
  
  if (data.direccion) partes.push(data.direccion)
  if (data.comuna) partes.push(data.comuna)
  if (data.ciudad) partes.push(data.ciudad)
  if (data.region) partes.push(data.region)
  
  return partes.join(', ')
}

/**
 * Obtiene el nombre de la empresa (razón social o nombre fantasía)
 */
export function obtenerNombreEmpresa(data: SimpleApiCompany): string {
  return data.razon_social || data.nombre_fantasia || ''
}
